#!/usr/bin/env python3
"""
RDF Diff - File comparison utility

Compares two files and outputs unified diff or JSON diff.
Replaces TUI diff panel for agent use.

Usage:
    rdf diff <file_a> <file_b> [--format unified|json]
"""

import argparse
import json
import sys
import difflib
from pathlib import Path

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from pipeline.cli_utils import success_response, error_response, ErrorCodes


def compute_diff(file_a: Path, file_b: Path) -> dict:
    """Compute diff between two files."""
    if not file_a.exists():
        raise FileNotFoundError(f"File not found: {file_a}")
    if not file_b.exists():
        raise FileNotFoundError(f"File not found: {file_b}")

    with open(file_a, 'r', encoding='utf-8') as f:
        lines_a = f.readlines()
    with open(file_b, 'r', encoding='utf-8') as f:
        lines_b = f.readlines()

    # Compute unified diff
    unified = list(difflib.unified_diff(
        lines_a, lines_b,
        fromfile=str(file_a),
        tofile=str(file_b),
        lineterm=''
    ))

    # Compute detailed changes
    matcher = difflib.SequenceMatcher(None, lines_a, lines_b)
    changes = []

    for tag, i1, i2, j1, j2 in matcher.get_opcodes():
        if tag == 'equal':
            continue

        change = {
            "type": tag,
            "file_a": {
                "start_line": i1 + 1,
                "end_line": i2,
                "content": ''.join(lines_a[i1:i2]) if i1 < i2 else ""
            },
            "file_b": {
                "start_line": j1 + 1,
                "end_line": j2,
                "content": ''.join(lines_b[j1:j2]) if j1 < j2 else ""
            }
        }
        changes.append(change)

    # Compute statistics
    added_lines = sum(1 for line in unified if line.startswith('+') and not line.startswith('+++'))
    removed_lines = sum(1 for line in unified if line.startswith('-') and not line.startswith('---'))

    return {
        "file_a": str(file_a),
        "file_b": str(file_b),
        "identical": len(changes) == 0,
        "statistics": {
            "lines_added": added_lines,
            "lines_removed": removed_lines,
            "total_changes": len(changes)
        },
        "unified_diff": '\n'.join(unified),
        "changes": changes
    }


def main():
    parser = argparse.ArgumentParser(
        description="Compare two files",
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    parser.add_argument("file_a", help="First file to compare")
    parser.add_argument("file_b", help="Second file to compare")
    parser.add_argument("--format", choices=["unified", "json"], default="unified",
                        help="Output format (default: unified)")

    args = parser.parse_args()

    try:
        file_a = Path(args.file_a)
        file_b = Path(args.file_b)

        result = compute_diff(file_a, file_b)

        if args.format == "json":
            response = success_response(
                f"Compared {file_a.name} with {file_b.name}",
                data=result
            )
            response.print_json()
        else:
            # Unified diff output
            if result["identical"]:
                print(f"Files are identical: {file_a} and {file_b}")
            else:
                print(result["unified_diff"])
                print(f"\n# {result['statistics']['lines_added']} additions, "
                      f"{result['statistics']['lines_removed']} deletions")

        return 0

    except FileNotFoundError as e:
        resp = error_response(
            ErrorCodes.FILE_NOT_FOUND,
            str(e)
        )
        if args.format == "json":
            resp.print_json()
        else:
            print(f"Error: {e}")
        return 1

    except Exception as e:
        resp = error_response(
            ErrorCodes.UNKNOWN_ERROR,
            str(e)
        )
        if args.format == "json":
            resp.print_json()
        else:
            print(f"Error: {e}")
        return 1


if __name__ == "__main__":
    sys.exit(main())
