#!/usr/bin/env python3
"""
RDF Context - Agent Session Warm-Start

Returns a "Context Packet" optimized for Claude Code to resume a session.
Single command replaces 5+ file reads for context reconstruction.

Usage:
    rdf context BOOK_xxx --format json
    rdf context --list
    rdf context  # Auto-detect most recent active project
"""

import argparse
import json
import sys
import logging
from pathlib import Path
from datetime import datetime
from typing import Dict, Any, Optional, List

# Setup path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))
sys.path.insert(0, str(Path(__file__).parent))

from pipeline.cli_utils import success_response, error_response, ErrorCodes

# Setup logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Phase names
PHASE_NAMES = {
    1: "Initial Research",
    2: "Gap Analysis",
    3: "Gap Filling",
    4: "Synthesis",
    5: "Draft Generation",
}


def find_project_directories() -> Dict[str, Path]:
    """Find all project directories (books and essays)."""
    projects = {}

    # Book projects
    books_dir = Path("projects/books")
    if books_dir.exists():
        for d in books_dir.iterdir():
            if d.is_dir() and d.name.startswith("BOOK_"):
                projects[d.name] = d

    # Essay projects
    essays_dir = Path("essays")
    if essays_dir.exists():
        for d in essays_dir.iterdir():
            if d.is_dir() and d.name.startswith("ESSAY_"):
                projects[d.name] = d

    return projects


def get_most_recent_project() -> Optional[str]:
    """Get the most recently modified project."""
    projects = find_project_directories()
    if not projects:
        return None

    # Find most recently modified
    latest = None
    latest_time = None

    for project_id, project_dir in projects.items():
        state_file = project_dir / "workflow_state.json"
        if state_file.exists():
            mtime = state_file.stat().st_mtime
            if latest_time is None or mtime > latest_time:
                latest_time = mtime
                latest = project_id

    return latest


def load_project_state(project_dir: Path) -> Dict[str, Any]:
    """Load project state from various files."""
    state = {}

    # Load workflow_state.json
    state_file = project_dir / "workflow_state.json"
    if state_file.exists():
        with open(state_file, 'r') as f:
            state['workflow'] = json.load(f)

    # Load project.json (for books)
    project_file = project_dir / "project.json"
    if project_file.exists():
        with open(project_file, 'r') as f:
            state['project'] = json.load(f)

    # Load checkpoint.json
    checkpoint_file = project_dir / "checkpoint.json"
    if checkpoint_file.exists():
        with open(checkpoint_file, 'r') as f:
            state['checkpoint'] = json.load(f)

    # Load session_log.json
    session_log = project_dir / "session_log.json"
    if session_log.exists():
        with open(session_log, 'r') as f:
            state['session_log'] = json.load(f)

    # Load gaps.md summary
    gaps_file = project_dir / "gaps.md"
    if gaps_file.exists():
        with open(gaps_file, 'r') as f:
            state['gaps_summary'] = f.read()[:2000]  # First 2000 chars

    return state


def get_queue_items(project_id: str) -> List[Dict[str, Any]]:
    """Get pending queue items for the project."""
    try:
        from review_queue import ReviewQueue
        queue = ReviewQueue()
        items = queue.list_pending()

        # Filter to project-relevant items
        relevant = []
        for item in items:
            if project_id in str(item.get('context', '')):
                relevant.append({
                    'id': item.get('item_id'),
                    'type': item.get('item_type'),
                    'query': item.get('query') or item.get('description', '')[:100],
                    'priority': item.get('priority', 'medium'),
                })
        return relevant
    except Exception as e:
        logger.warning(f"Could not load queue items: {e}")
        return []


def get_resource_status() -> Dict[str, Any]:
    """Get resource/credit status."""
    status = {}

    # Tavily credits
    try:
        from web_search import get_credit_tracker
        tracker = get_credit_tracker()
        if tracker:
            status['tavily_credits_remaining'] = tracker.remaining_credits
            status['tavily_credits_used'] = tracker.credits_used
    except Exception:
        pass

    # Cost tracking
    try:
        from config import GENERATION_CONFIG
        max_budget = GENERATION_CONFIG.get('max_budget_usd', 10.00)
        status['max_budget_usd'] = max_budget
        # TODO: Track actual spend
        status['budget_remaining_usd'] = max_budget
    except Exception:
        pass

    return status


def generate_context_packet(project_id: str) -> Dict[str, Any]:
    """
    Generate a complete context packet for agent warm-start.

    This is the main function that assembles all context needed
    for Claude Code to resume a session efficiently.
    """
    projects = find_project_directories()

    if project_id not in projects:
        return {
            'error': f'Project not found: {project_id}',
            'available_projects': list(projects.keys())
        }

    project_dir = projects[project_id]
    state = load_project_state(project_dir)

    # Determine project type
    is_book = project_id.startswith("BOOK_")
    is_essay = project_id.startswith("ESSAY_")

    # Build context packet
    packet = {
        'meta': {
            'project_id': project_id,
            'project_type': 'book' if is_book else 'essay',
            'project_dir': str(project_dir),
        },
        'position': {},
        'recent_activity': [],
        'active_context': {},
        'blocking_items': {},
        'next_actions': [],
        'resource_status': get_resource_status(),
    }

    # Extract from workflow state
    workflow = state.get('workflow', {})
    packet['meta']['title'] = workflow.get('title') or workflow.get('topic', 'Unknown')
    packet['meta']['created'] = workflow.get('created_at')
    packet['meta']['last_activity'] = workflow.get('updated_at') or workflow.get('created_at')

    # Position info
    if is_book:
        project_data = state.get('project', {})
        checkpoint = state.get('checkpoint', {})

        current_phase = checkpoint.get('phase') or project_data.get('current_phase', 1)
        packet['position'] = {
            'current_phase': current_phase,
            'phase_name': PHASE_NAMES.get(current_phase, f"Phase {current_phase}"),
            'phase_status': project_data.get('phase_status', {}),
        }

        # Chapter progress
        chapters = project_data.get('chapters', [])
        if chapters:
            chapter_idx = checkpoint.get('chapter_index', 0)
            if chapter_idx < len(chapters):
                current_chapter = chapters[chapter_idx]
                packet['position']['current_chapter'] = {
                    'number': current_chapter.get('chapter_number', chapter_idx + 1),
                    'title': current_chapter.get('title', 'Unknown'),
                    'status': current_chapter.get('status', 'pending'),
                }

            completed = sum(1 for c in chapters if c.get('status') == 'complete')
            packet['position']['progress'] = f"{completed}/{len(chapters)} chapters"

        # Subject progress
        subject_idx = checkpoint.get('subject_index', 0)
        total_subjects = sum(len(c.get('subjects', [])) for c in chapters)
        if total_subjects > 0:
            packet['position']['subject_progress'] = f"{subject_idx}/{total_subjects} subjects"

        # Active context - last synthesis
        for chapter in reversed(chapters):
            if chapter.get('research_summary'):
                packet['active_context']['last_synthesis'] = chapter['research_summary'][:500]
                break

        # Gaps
        all_gaps = project_data.get('all_gaps', [])
        pending_gaps = [g for g in all_gaps if not g.get('resolved')]
        if pending_gaps:
            packet['active_context']['pending_gaps'] = [
                {'query': g.get('query'), 'priority': g.get('priority', 'medium')}
                for g in pending_gaps[:5]
            ]

    elif is_essay:
        packet['position'] = {
            'current_phase': workflow.get('current_phase', 'initialized'),
            'phases_completed': workflow.get('phases_completed', []),
            'mode': workflow.get('mode', 'brief'),
        }

        # Essay brief context
        brief_file = project_dir / "essay_brief.json"
        if brief_file.exists():
            with open(brief_file, 'r') as f:
                brief = json.load(f)
            packet['active_context']['brief_summary'] = {
                'target_pages': brief.get('target_pages'),
                'style': brief.get('style_name'),
                'sections': brief.get('suggested_structure', {}).get('sections', []),
            }

        # Research context
        research_file = project_dir / "research.json"
        if research_file.exists():
            with open(research_file, 'r') as f:
                research = json.load(f)
            packet['active_context']['research_summary'] = research.get('synthesis', '')[:500]
            packet['active_context']['sources_count'] = len(research.get('sources', []))

    # Queue items (blocking)
    queue_items = get_queue_items(project_id)
    if queue_items:
        critical = [q for q in queue_items if q.get('priority') == 'critical' or q.get('priority') == 'high']
        packet['blocking_items'] = {
            'queue_items': len(queue_items),
            'critical': len(critical),
            'details': queue_items[:5],  # First 5 items
        }

    # Session log for recent activity
    session_log = state.get('session_log', {})
    if isinstance(session_log, dict) and 'entries' in session_log:
        entries = session_log['entries'][-5:]  # Last 5 entries
        packet['recent_activity'] = [
            e.get('message', str(e)) for e in entries
        ]

    # Generate next actions
    next_actions = []

    if packet['blocking_items'].get('queue_items', 0) > 0:
        next_actions.append(f"Review {packet['blocking_items']['queue_items']} pending queue items")

    if is_book:
        phase = packet['position'].get('current_phase', 1)
        if phase < 5:
            next_actions.append(f"Continue Phase {phase}: {PHASE_NAMES.get(phase)}")
        elif phase == 5:
            phase_status = packet['position'].get('phase_status', {})
            if phase_status.get(5) == 'briefs_ready':
                next_actions.append("Write chapter drafts from the generated briefs")
            else:
                next_actions.append("Complete draft generation")
    elif is_essay:
        mode = packet['position'].get('mode', 'brief')
        phase = packet['position'].get('current_phase', '')
        if mode == 'brief' and phase == 'brief_ready':
            next_actions.append("Write essay draft using the generated brief")
        elif phase == 'draft_complete':
            next_actions.append("Review and validate the generated draft")

    packet['next_actions'] = next_actions

    return packet


def list_projects() -> List[Dict[str, Any]]:
    """List all projects with basic info."""
    projects = find_project_directories()
    result = []

    for project_id, project_dir in projects.items():
        info = {
            'project_id': project_id,
            'type': 'book' if project_id.startswith('BOOK_') else 'essay',
            'path': str(project_dir),
        }

        # Load basic state
        state_file = project_dir / "workflow_state.json"
        if state_file.exists():
            with open(state_file, 'r') as f:
                state = json.load(f)
            info['title'] = state.get('title') or state.get('topic', 'Unknown')
            info['status'] = state.get('current_phase', 'unknown')
            info['last_modified'] = datetime.fromtimestamp(
                state_file.stat().st_mtime
            ).isoformat()

        result.append(info)

    # Sort by last modified (most recent first)
    result.sort(key=lambda x: x.get('last_modified', ''), reverse=True)
    return result


def main():
    parser = argparse.ArgumentParser(
        description="Agent session warm-start context loader",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Get context for specific project
  rdf context BOOK_xxx --format json

  # Auto-detect most recent project
  rdf context --format json

  # List all projects
  rdf context --list
        """
    )

    parser.add_argument(
        "project_id",
        nargs='?',
        help="Project ID (e.g., BOOK_xxx, ESSAY_xxx). Auto-detects if not specified."
    )
    parser.add_argument(
        "--list", "-l",
        action="store_true",
        help="List all projects"
    )
    parser.add_argument(
        "--format", "-f",
        choices=["json", "text"],
        default="json",
        help="Output format"
    )

    args = parser.parse_args()

    # List mode
    if args.list:
        projects = list_projects()
        if args.format == "json":
            print(json.dumps({'projects': projects}, indent=2))
        else:
            if not projects:
                print("No projects found.")
            else:
                print(f"\n{'='*60}")
                print("PROJECTS")
                print(f"{'='*60}")
                for p in projects:
                    print(f"\n{p['project_id']} ({p['type']})")
                    print(f"  Title: {p.get('title', 'Unknown')}")
                    print(f"  Status: {p.get('status', 'unknown')}")
                    print(f"  Modified: {p.get('last_modified', 'unknown')}")
        return 0

    # Determine project ID
    project_id = args.project_id
    if not project_id:
        project_id = get_most_recent_project()
        if not project_id:
            resp = error_response(
                ErrorCodes.NOT_FOUND,
                "No projects found. Create a book or essay first.",
            )
            if args.format == "json":
                resp.print_json()
            else:
                print("No projects found.")
            return 1

    # Generate context packet
    try:
        packet = generate_context_packet(project_id)

        if 'error' in packet:
            resp = error_response(
                ErrorCodes.NOT_FOUND,
                packet['error'],
                data={'available_projects': packet.get('available_projects', [])}
            )
            if args.format == "json":
                resp.print_json()
            else:
                print(f"Error: {packet['error']}")
            return 1

        response = success_response(
            f"Context loaded for {project_id}",
            data=packet
        )

        if args.format == "json":
            response.print_json()
        else:
            print(f"\n{'='*60}")
            print(f"CONTEXT: {packet['meta'].get('title', project_id)}")
            print(f"{'='*60}")
            print(f"Project ID: {project_id}")
            print(f"Type: {packet['meta'].get('project_type')}")
            print(f"Last Activity: {packet['meta'].get('last_activity')}")

            pos = packet.get('position', {})
            print(f"\nPosition:")
            print(f"  Phase: {pos.get('current_phase')} - {pos.get('phase_name', '')}")
            if pos.get('progress'):
                print(f"  Progress: {pos['progress']}")

            blocking = packet.get('blocking_items', {})
            if blocking.get('queue_items'):
                print(f"\nBlocking Items: {blocking['queue_items']} ({blocking.get('critical', 0)} critical)")

            if packet.get('next_actions'):
                print(f"\nNext Actions:")
                for action in packet['next_actions']:
                    print(f"  - {action}")

        return 0

    except Exception as e:
        logger.error(f"Context generation failed: {e}")
        resp = error_response(
            ErrorCodes.UNKNOWN_ERROR,
            str(e)
        )
        if args.format == "json":
            resp.print_json()
        else:
            print(f"Error: {e}")
        return 1


if __name__ == "__main__":
    sys.exit(main())
