#!/usr/bin/env python3
"""
Configuration Management for Research Development Framework.

Provides interactive and programmatic configuration for:
- Entity extraction (GLiNER vs OpenAI vs Hybrid)
- Relationship extraction settings
- Other configurable features

Usage:
    rdf config                              # Show all configuration status
    rdf config entity-extraction            # Configure entity extraction
    rdf config entity-extraction --provider hybrid --relation-backend openai
    rdf config show                         # Show current settings (JSON)
    rdf config check                        # Check what needs configuration
"""

import argparse
import json
import logging
import os
import sys
from pathlib import Path
from typing import Dict, Any, Optional, List

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent))

from cli_utils import success_response, error_response, format_output

# Setup logging
logging.basicConfig(level=logging.INFO, format='%(levelname)s: %(message)s')
logger = logging.getLogger(__name__)

# =============================================================================
# CONFIGURATION FILE MANAGEMENT
# =============================================================================

BASE_DIR = Path(__file__).resolve().parent.parent
CONFIG_PATH = BASE_DIR / 'config' / 'project.yaml'


def load_config() -> Dict[str, Any]:
    """Load the current project configuration."""
    try:
        import yaml
        if CONFIG_PATH.exists():
            with open(CONFIG_PATH, 'r') as f:
                return yaml.safe_load(f) or {}
    except ImportError:
        logger.error("PyYAML not installed. Run: pip install pyyaml")
    return {}


def save_config(config: Dict[str, Any]) -> bool:
    """Save configuration back to project.yaml."""
    try:
        import yaml

        # Read existing file to preserve comments structure
        if CONFIG_PATH.exists():
            with open(CONFIG_PATH, 'r') as f:
                existing_content = f.read()

        with open(CONFIG_PATH, 'w') as f:
            yaml.dump(config, f, default_flow_style=False, sort_keys=False, allow_unicode=True)

        return True
    except Exception as e:
        logger.error(f"Failed to save configuration: {e}")
        return False


def update_config_section(section: str, updates: Dict[str, Any]) -> bool:
    """Update a specific section of the configuration."""
    config = load_config()
    if section not in config:
        config[section] = {}
    config[section].update(updates)
    return save_config(config)


# =============================================================================
# DEPENDENCY CHECKING
# =============================================================================

def check_gliner_available() -> Dict[str, Any]:
    """Check if GLiNER is available."""
    try:
        import gliner
        return {
            'available': True,
            'installed': True,
            'message': 'GLiNER is installed and ready'
        }
    except ImportError:
        return {
            'available': False,
            'installed': False,
            'message': 'GLiNER not installed',
            'install_command': 'pip install gliner'
        }


def check_openai_available() -> Dict[str, Any]:
    """Check if OpenAI is available and configured."""
    try:
        import openai
        api_key = os.getenv('OPENAI_API_KEY', '').strip()
        if api_key and not api_key.startswith('your_'):
            return {
                'available': True,
                'installed': True,
                'api_key_set': True,
                'message': 'OpenAI is installed and API key is configured'
            }
        else:
            return {
                'available': False,
                'installed': True,
                'api_key_set': False,
                'message': 'OpenAI installed but OPENAI_API_KEY not set',
                'setup_command': 'Add OPENAI_API_KEY to .env file'
            }
    except ImportError:
        return {
            'available': False,
            'installed': False,
            'api_key_set': False,
            'message': 'OpenAI not installed',
            'install_command': 'pip install openai'
        }


def check_llama_available() -> Dict[str, Any]:
    """Check if LLaMA (llama-cpp-python) is available."""
    try:
        import llama_cpp
        model_path = os.getenv('LLAMA_MODEL_PATH')
        if model_path and Path(model_path).exists():
            return {
                'available': True,
                'installed': True,
                'model_configured': True,
                'message': f'LLaMA installed with model at {model_path}'
            }
        else:
            return {
                'available': False,
                'installed': True,
                'model_configured': False,
                'message': 'LLaMA installed but no model configured',
                'setup_command': 'Set LLAMA_MODEL_PATH to your .gguf model file'
            }
    except ImportError:
        return {
            'available': False,
            'installed': False,
            'model_configured': False,
            'message': 'llama-cpp-python not installed',
            'install_command': 'pip install llama-cpp-python'
        }


def check_all_extractors() -> Dict[str, Any]:
    """Check availability of all entity extractors."""
    return {
        'gliner': check_gliner_available(),
        'openai': check_openai_available(),
        'llama': check_llama_available(),
    }


# =============================================================================
# CONFIGURATION STATUS
# =============================================================================

def get_configuration_status() -> Dict[str, Any]:
    """Get complete configuration status for Claude Code."""
    config = load_config()
    extractors = check_all_extractors()

    entity_config = config.get('entity_extraction', {})
    is_configured = entity_config.get('configured', False)
    current_provider = entity_config.get('provider', 'gliner')

    # Determine what's available
    available_providers = []
    if extractors['gliner']['available']:
        available_providers.append('gliner')
    if extractors['openai']['available']:
        available_providers.append('openai')
        available_providers.append('hybrid')  # Hybrid requires OpenAI for relations
    if extractors['llama']['available']:
        if 'hybrid' not in available_providers:
            available_providers.append('hybrid')

    # Check if current provider is actually available
    provider_available = current_provider in available_providers or (
        current_provider == 'hybrid' and
        extractors['gliner']['available'] and
        (extractors['openai']['available'] or extractors['llama']['available'])
    )

    return {
        'entity_extraction': {
            'configured': is_configured,
            'provider': current_provider,
            'provider_available': provider_available,
            'relation_backend': entity_config.get('relation_backend', 'openai'),
            'openai_model': entity_config.get('openai_model', 'gpt-4o-mini'),
            'extract_relations': entity_config.get('extract_relations', False),
        },
        'available_providers': available_providers,
        'extractor_status': extractors,
        'needs_configuration': not is_configured,
        'recommendations': get_recommendations(extractors, is_configured, current_provider)
    }


def get_recommendations(extractors: Dict, is_configured: bool, current_provider: str) -> List[str]:
    """Generate configuration recommendations."""
    recommendations = []

    if not is_configured:
        recommendations.append("Entity extraction has not been configured. Run: rdf config entity-extraction")

    # Recommend hybrid if OpenAI is available but not being used
    if extractors['openai']['available'] and current_provider == 'gliner':
        recommendations.append(
            "OpenAI is available. Consider 'hybrid' mode for higher quality relationship extraction."
        )

    # Warn if current provider isn't available
    if current_provider == 'openai' and not extractors['openai']['available']:
        recommendations.append(
            "OpenAI provider selected but not available. Configure API key or switch to 'gliner'."
        )

    if current_provider == 'hybrid':
        if not extractors['gliner']['available']:
            recommendations.append("Hybrid mode requires GLiNER. Install with: pip install gliner")
        if not extractors['openai']['available'] and not extractors['llama']['available']:
            recommendations.append("Hybrid mode requires OpenAI or LLaMA for relationship extraction.")

    return recommendations


def get_unconfigured_features() -> List[Dict[str, Any]]:
    """Get list of features that need configuration."""
    config = load_config()
    unconfigured = []

    # Check entity extraction
    entity_config = config.get('entity_extraction', {})
    if not entity_config.get('configured', False):
        extractors = check_all_extractors()
        unconfigured.append({
            'feature': 'entity_extraction',
            'name': 'Entity Extraction (GraphRAG)',
            'description': 'Configure how entities and relationships are extracted for the knowledge graph',
            'configure_command': 'rdf config entity-extraction',
            'options': [
                {
                    'id': 'gliner',
                    'label': 'GLiNER (Fast/Free)',
                    'description': 'CPU-based entity extraction. Fast and free but no relationship extraction.',
                    'available': extractors['gliner']['available'],
                    'install': extractors['gliner'].get('install_command')
                },
                {
                    'id': 'openai',
                    'label': 'OpenAI (Quality)',
                    'description': 'Cloud-based extraction with GPT-4. Highest quality entities and relationships.',
                    'available': extractors['openai']['available'],
                    'install': extractors['openai'].get('install_command') or extractors['openai'].get('setup_command')
                },
                {
                    'id': 'hybrid',
                    'label': 'Hybrid (Recommended)',
                    'description': 'GLiNER for fast entity detection + OpenAI for high-quality relationships.',
                    'available': extractors['gliner']['available'] and extractors['openai']['available'],
                    'recommended': True
                }
            ]
        })

    return unconfigured


# =============================================================================
# CONFIGURATION COMMANDS
# =============================================================================

def configure_entity_extraction(
    provider: str,
    relation_backend: str = 'openai',
    openai_model: str = 'gpt-4o-mini',
    extract_relations: bool = None,
    gliner_model_size: str = 'medium',
    confidence_threshold: float = 0.4
) -> Dict[str, Any]:
    """Configure entity extraction settings."""

    # Validate provider
    valid_providers = ['gliner', 'openai', 'hybrid']
    if provider not in valid_providers:
        return error_response(
            message=f"Invalid provider: {provider}",
            code="INVALID_PROVIDER",
            data={'valid_options': valid_providers}
        )

    # Check if provider is available
    extractors = check_all_extractors()

    if provider == 'gliner' and not extractors['gliner']['available']:
        return error_response(
            message="GLiNER is not installed",
            code="DEPENDENCY_MISSING",
            data=extractors['gliner'],
            actionable_advice="Install with: pip install gliner"
        )

    if provider == 'openai' and not extractors['openai']['available']:
        return error_response(
            message="OpenAI is not available",
            code="DEPENDENCY_MISSING",
            data=extractors['openai'],
            actionable_advice=extractors['openai'].get('install_command') or extractors['openai'].get('setup_command')
        )

    if provider == 'hybrid':
        if not extractors['gliner']['available']:
            return error_response(
                message="Hybrid mode requires GLiNER for entity extraction",
                code="DEPENDENCY_MISSING",
                data=extractors['gliner'],
                actionable_advice="Install with: pip install gliner"
            )

        # Check relation backend
        if relation_backend == 'openai' and not extractors['openai']['available']:
            return error_response(
                message="Hybrid mode with OpenAI relations requires OpenAI API key",
                code="DEPENDENCY_MISSING",
                data=extractors['openai'],
                actionable_advice="Add OPENAI_API_KEY to .env file"
            )

        if relation_backend == 'llama' and not extractors['llama']['available']:
            return error_response(
                message="Hybrid mode with LLaMA relations requires llama-cpp-python",
                code="DEPENDENCY_MISSING",
                data=extractors['llama'],
                actionable_advice="Install with: pip install llama-cpp-python"
            )

    # Determine if relations should be extracted
    if extract_relations is None:
        # Default: enable for openai and hybrid, disable for gliner-only
        extract_relations = provider in ['openai', 'hybrid']

    # Build configuration
    new_config = {
        'configured': True,
        'provider': provider,
        'relation_backend': relation_backend,
        'openai_model': openai_model,
        'gliner_model_size': gliner_model_size,
        'confidence_threshold': confidence_threshold,
        'extract_relations': extract_relations,
    }

    # Save configuration
    if update_config_section('entity_extraction', new_config):
        return success_response(
            message=f"Entity extraction configured: {provider}",
            code="CONFIGURATION_SAVED",
            data={
                'provider': provider,
                'relation_backend': relation_backend if provider == 'hybrid' else None,
                'openai_model': openai_model if provider in ['openai', 'hybrid'] else None,
                'extract_relations': extract_relations,
                'gliner_model_size': gliner_model_size if provider in ['gliner', 'hybrid'] else None,
            }
        )
    else:
        return error_response(
            message="Failed to save configuration",
            code="SAVE_FAILED"
        )


# =============================================================================
# CLI INTERFACE
# =============================================================================

def main():
    parser = argparse.ArgumentParser(
        description='RDF Configuration Management',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
    # Show configuration status
    rdf config
    rdf config show --format json

    # Check what needs configuration (for Claude Code)
    rdf config check --format json

    # Configure entity extraction interactively
    rdf config entity-extraction

    # Configure entity extraction with specific settings
    rdf config entity-extraction --provider hybrid --relation-backend openai

    # Quick setup with recommended settings
    rdf config entity-extraction --provider hybrid --openai-model gpt-4o
"""
    )

    parser.add_argument('action', nargs='?', default='show',
                        choices=['show', 'check', 'entity-extraction'],
                        help='Configuration action')
    parser.add_argument('--format', choices=['text', 'json'], default='text',
                        help='Output format')

    # Entity extraction options
    parser.add_argument('--provider', choices=['gliner', 'openai', 'hybrid'],
                        help='Entity extraction provider')
    parser.add_argument('--relation-backend', choices=['openai', 'llama'],
                        default='openai', help='Backend for relationship extraction (hybrid mode)')
    parser.add_argument('--openai-model', default='gpt-4o-mini',
                        help='OpenAI model for extraction')
    parser.add_argument('--extract-relations', action='store_true',
                        help='Enable relationship extraction')
    parser.add_argument('--no-relations', action='store_true',
                        help='Disable relationship extraction')
    parser.add_argument('--gliner-model-size', choices=['small', 'medium', 'large', 'multi'],
                        default='medium', help='GLiNER model size')
    parser.add_argument('--confidence-threshold', type=float, default=0.4,
                        help='Confidence threshold for entity detection')

    args = parser.parse_args()

    # Handle actions
    if args.action == 'show':
        status = get_configuration_status()

        if args.format == 'json':
            response = success_response(
                message="Configuration status",
                code="CONFIG_STATUS",
                data=status
            )
            print(json.dumps(response, indent=2))
        else:
            print_status_text(status)

    elif args.action == 'check':
        unconfigured = get_unconfigured_features()
        status = get_configuration_status()

        if unconfigured:
            response = {
                'status': 'warning',
                'code': 'CONFIGURATION_REQUIRED',
                'message': f"{len(unconfigured)} feature(s) need configuration",
                'data': {
                    'unconfigured_features': unconfigured,
                    'current_status': status
                },
                'decision_packet': {
                    'decision_id': 'entity_extraction_provider',
                    'question': 'Which entity extraction provider should be used for GraphRAG?',
                    'options': unconfigured[0]['options'] if unconfigured else [],
                    'recommended': 'hybrid'
                }
            }
        else:
            response = success_response(
                message="All features are configured",
                code="FULLY_CONFIGURED",
                data=status
            )

        if args.format == 'json':
            print(json.dumps(response, indent=2))
        else:
            if unconfigured:
                print("\n⚠️  CONFIGURATION REQUIRED")
                print("=" * 50)
                for feature in unconfigured:
                    print(f"\n{feature['name']}")
                    print(f"  {feature['description']}")
                    print(f"  Configure with: {feature['configure_command']}")
                    print("\n  Options:")
                    for opt in feature['options']:
                        status_icon = "✓" if opt.get('available') else "✗"
                        rec = " (RECOMMENDED)" if opt.get('recommended') else ""
                        print(f"    [{status_icon}] {opt['id']}: {opt['label']}{rec}")
                        if not opt.get('available') and opt.get('install'):
                            print(f"        Install: {opt['install']}")
            else:
                print("\n✓ All features are configured")

    elif args.action == 'entity-extraction':
        if args.provider:
            # Direct configuration with provided options
            extract_relations = None
            if args.extract_relations:
                extract_relations = True
            elif args.no_relations:
                extract_relations = False

            result = configure_entity_extraction(
                provider=args.provider,
                relation_backend=args.relation_backend,
                openai_model=args.openai_model,
                extract_relations=extract_relations,
                gliner_model_size=args.gliner_model_size,
                confidence_threshold=args.confidence_threshold
            )

            if args.format == 'json':
                print(json.dumps(result, indent=2))
            else:
                if result.get('status') == 'success':
                    print(f"\n✓ {result['message']}")
                    data = result.get('data', {})
                    print(f"  Provider: {data.get('provider')}")
                    if data.get('relation_backend'):
                        print(f"  Relation Backend: {data.get('relation_backend')}")
                    if data.get('openai_model'):
                        print(f"  OpenAI Model: {data.get('openai_model')}")
                    print(f"  Extract Relations: {data.get('extract_relations')}")
                else:
                    print(f"\n✗ Error: {result['message']}")
                    if result.get('actionable_advice'):
                        print(f"  → {result['actionable_advice']}")
        else:
            # Show current status and prompt for configuration
            status = get_configuration_status()
            extractors = status['extractor_status']

            if args.format == 'json':
                response = {
                    'status': 'prompt',
                    'code': 'AWAITING_SELECTION',
                    'message': 'Select entity extraction provider',
                    'data': status,
                    'decision_packet': {
                        'decision_id': 'entity_extraction_provider',
                        'question': 'Which entity extraction provider should be used?',
                        'options': [
                            {
                                'id': 'gliner',
                                'label': 'GLiNER (Fast/Free)',
                                'available': extractors['gliner']['available'],
                                'description': 'CPU-based, no API costs. Entities only.'
                            },
                            {
                                'id': 'openai',
                                'label': 'OpenAI (Quality)',
                                'available': extractors['openai']['available'],
                                'description': 'Cloud-based, best quality. ~$0.01/1K tokens.'
                            },
                            {
                                'id': 'hybrid',
                                'label': 'Hybrid (Recommended)',
                                'available': extractors['gliner']['available'] and extractors['openai']['available'],
                                'description': 'GLiNER entities + OpenAI relations. Best balance.',
                                'recommended': True
                            }
                        ],
                        'default': 'hybrid' if (extractors['gliner']['available'] and extractors['openai']['available']) else 'gliner'
                    }
                }
                print(json.dumps(response, indent=2))
            else:
                print("\n" + "=" * 60)
                print("ENTITY EXTRACTION CONFIGURATION")
                print("=" * 60)
                print("\nAvailable Providers:\n")

                for name, info in extractors.items():
                    status_icon = "✓" if info['available'] else "✗"
                    print(f"  [{status_icon}] {name.upper()}")
                    print(f"      {info['message']}")
                    if not info['available'] and info.get('install_command'):
                        print(f"      Install: {info['install_command']}")
                    if not info['available'] and info.get('setup_command'):
                        print(f"      Setup: {info['setup_command']}")

                print("\n" + "-" * 60)
                print("\nTo configure, run:")
                print("  rdf config entity-extraction --provider <choice>")
                print("\nRecommended:")
                if extractors['gliner']['available'] and extractors['openai']['available']:
                    print("  rdf config entity-extraction --provider hybrid")
                elif extractors['openai']['available']:
                    print("  rdf config entity-extraction --provider openai")
                else:
                    print("  rdf config entity-extraction --provider gliner")


def print_status_text(status: Dict[str, Any]):
    """Print configuration status in human-readable format."""
    print("\n" + "=" * 60)
    print("RDF CONFIGURATION STATUS")
    print("=" * 60)

    # Entity Extraction
    ee = status['entity_extraction']
    print("\nENTITY EXTRACTION (GraphRAG)")
    print("-" * 40)

    configured_icon = "✓" if ee['configured'] else "✗"
    print(f"  Configured:        [{configured_icon}] {'Yes' if ee['configured'] else 'No - run: rdf config entity-extraction'}")
    print(f"  Provider:          {ee['provider']}")

    if ee['provider'] == 'hybrid':
        print(f"  Relation Backend:  {ee['relation_backend']}")

    if ee['provider'] in ['openai', 'hybrid']:
        print(f"  OpenAI Model:      {ee['openai_model']}")

    print(f"  Extract Relations: {'Yes' if ee['extract_relations'] else 'No'}")

    available_icon = "✓" if ee['provider_available'] else "✗"
    print(f"  Provider Ready:    [{available_icon}] {'Yes' if ee['provider_available'] else 'No - check dependencies'}")

    # Extractor Status
    print("\nEXTRACTOR AVAILABILITY")
    print("-" * 40)
    for name, info in status['extractor_status'].items():
        icon = "✓" if info['available'] else "✗"
        print(f"  [{icon}] {name}: {info['message']}")

    # Recommendations
    if status['recommendations']:
        print("\nRECOMMENDATIONS")
        print("-" * 40)
        for rec in status['recommendations']:
            print(f"  → {rec}")

    print("=" * 60 + "\n")


if __name__ == '__main__':
    main()
