#!/usr/bin/env python3
"""
Book Workflow Phase Executors

Implements the 5-phase research workflow:
- Phase 1: Initial Research (search library for each subject)
- Phase 2: Gap Analysis (aggregate and prioritize gaps)
- Phase 3: Gap Filling (manual review or auto-Tavily)
- Phase 4: Synthesis (compile research into summaries)
- Phase 5: Draft Generation (create chapter drafts)

Each phase is a separate executor class that can be run independently
or as part of the full workflow.
"""

import logging
from pathlib import Path
from datetime import datetime
from typing import List, Dict, Any, Optional, Callable
from abc import ABC, abstractmethod

from book_workflow_models import (
    BookProject, ChapterProject, SubjectResearch,
    WorkflowCheckpoint, IdentifiedGap
)

# Setup logging
logger = logging.getLogger(__name__)

# Import research components
try:
    from research_agent import ResearchAgent, ResearchSession, format_markdown_report
    AGENT_AVAILABLE = True
except ImportError:
    AGENT_AVAILABLE = False
    logger.warning("ResearchAgent not available")

try:
    from web_search import (
        TavilySearchInterface, get_credit_tracker,
        estimate_gap_search_cost, TAVILY_ENABLED
    )
    WEB_SEARCH_AVAILABLE = TAVILY_ENABLED
except ImportError:
    WEB_SEARCH_AVAILABLE = False
    logger.warning("Web search not available")

try:
    from db_utils import hybrid_search_with_rerank
    SEARCH_AVAILABLE = True
except ImportError:
    SEARCH_AVAILABLE = False


# =============================================================================
# BASE PHASE EXECUTOR
# =============================================================================

class PhaseExecutor(ABC):
    """Base class for phase executors."""

    phase_number: int = 0
    phase_name: str = "Base"

    def __init__(
        self,
        project: BookProject,
        checkpoint: WorkflowCheckpoint,
        progress_callback: Optional[Callable[[str, float], None]] = None,
        dry_run: bool = False,
        verbose: bool = False,
    ):
        """
        Initialize phase executor.

        Args:
            project: BookProject to operate on
            checkpoint: WorkflowCheckpoint for resume capability
            progress_callback: Callback for progress updates (message, progress 0-1)
            dry_run: If True, preview actions without executing
            verbose: If True, show detailed output
        """
        self.project = project
        self.checkpoint = checkpoint
        self.progress_callback = progress_callback
        self.dry_run = dry_run
        self.verbose = verbose

    def report_progress(self, message: str, progress: float = 0.0) -> None:
        """Report progress to callback."""
        if self.progress_callback:
            self.progress_callback(message, progress)
        if self.verbose:
            logger.info(f"[{self.phase_name}] {message} ({progress:.0%})")

    def save_checkpoint(self, chapter_index: int, subject_index: int) -> None:
        """Save checkpoint after each operation."""
        self.checkpoint.update(chapter_index, subject_index)
        project_dir = Path(self.project.project_dir)
        self.checkpoint.save(project_dir)

    @abstractmethod
    def execute(self) -> Dict[str, Any]:
        """Execute the phase. Returns status dict."""
        pass

    @abstractmethod
    def can_execute(self) -> tuple[bool, str]:
        """Check if phase can be executed. Returns (can_run, reason)."""
        pass


# =============================================================================
# PHASE 1: INITIAL RESEARCH
# =============================================================================

class Phase1Executor(PhaseExecutor):
    """Phase 1: Initial Research - Search library for each subject."""

    phase_number = 1
    phase_name = "Initial Research"

    def __init__(
        self,
        project: BookProject,
        checkpoint: WorkflowCheckpoint,
        max_iterations: int = 5,
        use_graphrag: bool = True,
        use_rerank: bool = True,
        **kwargs
    ):
        super().__init__(project, checkpoint, **kwargs)
        self.max_iterations = max_iterations
        self.use_graphrag = use_graphrag
        self.use_rerank = use_rerank

        # Initialize research agent
        if AGENT_AVAILABLE:
            self.agent = ResearchAgent(
                max_iterations=max_iterations,
                use_graphrag=use_graphrag,
                use_rerank=use_rerank,
                auto_fill_gaps=False,  # Collect gaps for Phase 2
            )
        else:
            self.agent = None

    def can_execute(self) -> tuple[bool, str]:
        if not AGENT_AVAILABLE:
            return False, "ResearchAgent not available"
        if not self.project.chapters:
            return False, "No chapters in project"
        return True, "Ready"

    def execute(self) -> Dict[str, Any]:
        """Execute Phase 1: Research each subject in each chapter."""
        can_run, reason = self.can_execute()
        if not can_run:
            return {'success': False, 'error': reason}

        self.project.set_phase(1, 'in_progress')

        # Resume from checkpoint if applicable
        start_chapter = self.checkpoint.chapter_index
        start_subject = self.checkpoint.subject_index

        total_subjects = self.project.total_subjects
        completed_subjects = 0

        # Count already completed
        for i, chapter in enumerate(self.project.chapters):
            if i < start_chapter:
                completed_subjects += len(chapter.get_completed_subjects())
            elif i == start_chapter:
                completed_subjects += start_subject

        results = {
            'chapters_processed': 0,
            'subjects_researched': 0,
            'total_chunks_found': 0,
            'total_gaps_found': 0,
            'errors': [],
        }

        for chapter_idx, chapter in enumerate(self.project.chapters):
            if chapter_idx < start_chapter:
                results['chapters_processed'] += 1
                continue

            chapter.update_status('researching')

            for subject_idx, subject_research in enumerate(chapter.subject_research):
                # Skip already completed or if resuming
                if chapter_idx == start_chapter and subject_idx < start_subject:
                    continue

                if subject_research.status == 'completed':
                    completed_subjects += 1
                    continue

                # Calculate progress
                progress = completed_subjects / total_subjects if total_subjects > 0 else 0
                self.report_progress(
                    f"Chapter {chapter.chapter_number}: {subject_research.subject}",
                    progress
                )

                if self.dry_run:
                    logger.info(f"[DRY RUN] Would research: {subject_research.subject}")
                    subject_research.status = 'completed'
                    completed_subjects += 1
                    continue

                # Execute research
                try:
                    subject_research.status = 'in_progress'
                    subject_research.started_at = datetime.now().isoformat()

                    # Run research agent
                    session = self.agent.research(subject_research.subject)

                    # Store results
                    subject_research.chunks_found = session.total_chunks
                    subject_research.documents_found = session.unique_documents
                    subject_research.synthesis = session.synthesis

                    # Convert gaps to dict format
                    subject_research.gaps = [
                        {
                            'description': g.description,
                            'suggested_query': g.suggested_query,
                            'identified_at': g.identified_at,
                            'searched': False,
                        }
                        for g in session.identified_gaps
                    ]

                    subject_research.status = 'completed'
                    subject_research.completed_at = datetime.now().isoformat()

                    results['subjects_researched'] += 1
                    results['total_chunks_found'] += session.total_chunks
                    results['total_gaps_found'] += len(session.identified_gaps)

                    # Reset agent for next subject
                    self.agent.all_results = {}
                    self.agent.search_history = []
                    self.agent.identified_gaps = []

                except Exception as e:
                    logger.error(f"Research failed for '{subject_research.subject}': {e}")
                    subject_research.status = 'failed'
                    results['errors'].append({
                        'chapter': chapter.chapter_number,
                        'subject': subject_research.subject,
                        'error': str(e),
                    })

                # Save checkpoint after each subject
                completed_subjects += 1
                self.save_checkpoint(chapter_idx, subject_idx + 1)

                # Save project state periodically
                self.project.save()

            # Chapter complete
            if all(sr.status in ['completed', 'failed'] for sr in chapter.subject_research):
                chapter.collect_gaps()
                chapter.update_status('gaps_identified')
                results['chapters_processed'] += 1

        # Phase complete
        self.project.set_phase(1, 'completed')
        self.project.collect_all_gaps()
        self.project.save()

        self.report_progress("Phase 1 complete", 1.0)
        return {'success': True, **results}


# =============================================================================
# PHASE 2: GAP ANALYSIS
# =============================================================================

class Phase2Executor(PhaseExecutor):
    """Phase 2: Gap Analysis - Aggregate and prioritize research gaps."""

    phase_number = 2
    phase_name = "Gap Analysis"

    def __init__(
        self,
        project: BookProject,
        checkpoint: WorkflowCheckpoint,
        priority_threshold: int = 2,
        **kwargs
    ):
        super().__init__(project, checkpoint, **kwargs)
        self.priority_threshold = priority_threshold

    def can_execute(self) -> tuple[bool, str]:
        if self.project.phase_status.get(1) != 'completed':
            return False, "Phase 1 not completed"
        return True, "Ready"

    def execute(self) -> Dict[str, Any]:
        """Execute Phase 2: Analyze and prioritize gaps."""
        can_run, reason = self.can_execute()
        if not can_run:
            return {'success': False, 'error': reason}

        self.project.set_phase(2, 'in_progress')
        self.report_progress("Analyzing gaps", 0.0)

        # Collect all gaps from chapters
        all_gaps = self.project.collect_all_gaps()

        # Deduplicate and prioritize
        gap_counts = {}  # query -> count
        gap_details = {}  # query -> gap info

        for gap in all_gaps:
            query = gap.get('suggested_query', '').lower().strip()
            if not query:
                continue

            if query in gap_counts:
                gap_counts[query] += 1
                # Merge chapter references
                existing = gap_details[query]
                chapters = set(existing.get('chapters', []))
                chapters.add(gap.get('chapter_number', 0))
                existing['chapters'] = list(chapters)
            else:
                gap_counts[query] = 1
                gap_details[query] = {
                    **gap,
                    'chapters': [gap.get('chapter_number', 0)],
                    'occurrence_count': 1,
                }

        # Update occurrence counts and calculate priority
        for query, gap in gap_details.items():
            gap['occurrence_count'] = gap_counts[query]
            gap['priority'] = 'high' if gap_counts[query] >= self.priority_threshold else 'medium'

        # Sort by occurrence count (high priority first)
        prioritized_gaps = sorted(
            gap_details.values(),
            key=lambda g: (-g.get('occurrence_count', 0), g.get('suggested_query', ''))
        )

        # Update project with prioritized gaps
        self.project.all_gaps = prioritized_gaps

        self.report_progress("Gap analysis complete", 1.0)

        # Calculate stats
        high_priority = sum(1 for g in prioritized_gaps if g.get('priority') == 'high')
        medium_priority = len(prioritized_gaps) - high_priority

        results = {
            'success': True,
            'total_gaps': len(prioritized_gaps),
            'high_priority': high_priority,
            'medium_priority': medium_priority,
            'unique_gaps': len(prioritized_gaps),
            'raw_gaps': len(all_gaps),
        }

        self.project.set_phase(2, 'completed')
        self.project.save()

        return results


# =============================================================================
# PHASE 3: GAP FILLING
# =============================================================================

class Phase3Executor(PhaseExecutor):
    """Phase 3: Gap Filling - Search web for gaps (manual or auto)."""

    phase_number = 3
    phase_name = "Gap Filling"

    def __init__(
        self,
        project: BookProject,
        checkpoint: WorkflowCheckpoint,
        auto_fill: bool = False,
        tavily_budget: int = 50,
        approved_gaps: Optional[List[str]] = None,
        **kwargs
    ):
        super().__init__(project, checkpoint, **kwargs)
        self.auto_fill = auto_fill
        self.tavily_budget = tavily_budget
        self.approved_gaps = approved_gaps or []

        if WEB_SEARCH_AVAILABLE:
            self.web_search = TavilySearchInterface()
            self.credit_tracker = get_credit_tracker()
        else:
            self.web_search = None
            self.credit_tracker = None

    def can_execute(self) -> tuple[bool, str]:
        if self.project.phase_status.get(2) != 'completed':
            return False, "Phase 2 not completed"
        if not WEB_SEARCH_AVAILABLE:
            return False, "Tavily web search not available (check TAVILY_API_KEY)"
        return True, "Ready"

    def get_gaps_to_fill(self) -> List[Dict[str, Any]]:
        """Get list of gaps to fill based on mode."""
        unfilled_gaps = [g for g in self.project.all_gaps if not g.get('searched', False)]

        if self.auto_fill:
            # Auto mode: fill all within budget
            return unfilled_gaps
        elif self.approved_gaps:
            # Manual mode with approved list
            return [g for g in unfilled_gaps if g.get('suggested_query') in self.approved_gaps]
        else:
            # No gaps approved
            return []

    def execute(self) -> Dict[str, Any]:
        """Execute Phase 3: Fill gaps via web search."""
        can_run, reason = self.can_execute()
        if not can_run:
            return {'success': False, 'error': reason}

        gaps_to_fill = self.get_gaps_to_fill()

        if not gaps_to_fill:
            logger.info("No gaps to fill (none approved or all already searched)")
            self.project.set_phase(3, 'completed')
            self.project.save()
            return {
                'success': True,
                'gaps_filled': 0,
                'credits_used': 0,
                'message': 'No gaps to fill',
            }

        self.project.set_phase(3, 'in_progress')

        # Check budget
        cost_estimate = estimate_gap_search_cost(len(gaps_to_fill))
        if cost_estimate['estimated_credits'] > self.tavily_budget:
            # Limit gaps to budget
            max_gaps = self.tavily_budget // 2  # Assuming advanced search = 2 credits
            gaps_to_fill = gaps_to_fill[:max_gaps]
            logger.warning(f"Limited to {max_gaps} gaps due to budget ({self.tavily_budget} credits)")

        results = {
            'success': True,
            'gaps_filled': 0,
            'credits_used': 0,
            'web_sources_added': 0,
            'errors': [],
        }

        initial_credits = self.credit_tracker.usage.total_used if self.credit_tracker else 0

        for i, gap in enumerate(gaps_to_fill):
            progress = i / len(gaps_to_fill) if gaps_to_fill else 0
            query = gap.get('suggested_query', '')
            self.report_progress(f"Searching: {query[:50]}...", progress)

            if self.dry_run:
                logger.info(f"[DRY RUN] Would search: {query}")
                gap['searched'] = True
                results['gaps_filled'] += 1
                continue

            try:
                # Execute search
                response = self.web_search.search(query, max_results=5)

                if response.error:
                    logger.warning(f"Search error for '{query}': {response.error}")
                    results['errors'].append({'query': query, 'error': response.error})
                    continue

                # Mark gap as searched
                gap['searched'] = True
                gap['search_results_count'] = len(response.results)
                results['gaps_filled'] += 1

                # Add web sources to appropriate chapter(s)
                chapters = gap.get('chapters', [])
                for chapter_num in chapters:
                    chapter = self.project.get_chapter(chapter_num)
                    if chapter:
                        for result in response.results:
                            web_source = {
                                'url': result.url,
                                'title': result.title,
                                'content': result.content[:1000],
                                'query': query,
                                'score': result.score,
                                'gap_description': gap.get('description', ''),
                                'added_at': datetime.now().isoformat(),
                            }
                            chapter.web_sources.append(web_source)
                            results['web_sources_added'] += 1

            except Exception as e:
                logger.error(f"Gap fill failed for '{query}': {e}")
                results['errors'].append({'query': query, 'error': str(e)})

        # Calculate credits used
        if self.credit_tracker:
            results['credits_used'] = self.credit_tracker.usage.total_used - initial_credits

        self.project.collect_all_sources()
        self.project.set_phase(3, 'completed')
        self.project.save()

        self.report_progress("Gap filling complete", 1.0)
        return results


# =============================================================================
# PHASE 4: SYNTHESIS
# =============================================================================

class Phase4Executor(PhaseExecutor):
    """Phase 4: Synthesis - Compile research into chapter summaries."""

    phase_number = 4
    phase_name = "Synthesis"

    def __init__(
        self,
        project: BookProject,
        checkpoint: WorkflowCheckpoint,
        **kwargs
    ):
        super().__init__(project, checkpoint, **kwargs)

        # Initialize LLM interface
        try:
            from research_agent import LLMInterface
            self.llm = LLMInterface()
        except ImportError:
            self.llm = None

    def can_execute(self) -> tuple[bool, str]:
        # Can run after Phase 2 (gaps optional)
        if self.project.phase_status.get(2) != 'completed':
            return False, "Phase 2 not completed"
        return True, "Ready"

    def synthesize_chapter(self, chapter: ChapterProject) -> str:
        """Generate research summary for a chapter."""
        if not self.llm:
            # Fallback: concatenate subject syntheses
            summaries = []
            for sr in chapter.subject_research:
                if sr.synthesis:
                    summaries.append(f"### {sr.subject}\n\n{sr.synthesis}")
            return "\n\n".join(summaries)

        # Build context from subject research and web sources
        context_parts = []
        for sr in chapter.subject_research:
            if sr.synthesis:
                context_parts.append(f"Subject: {sr.subject}\n{sr.synthesis[:2000]}")

        for ws in chapter.web_sources[:10]:
            context_parts.append(f"Web Source: {ws.get('title', 'Unknown')}\n{ws.get('content', '')[:500]}")

        context = "\n\n---\n\n".join(context_parts)

        prompt = f"""You are synthesizing research for a book chapter.

Chapter Title: {chapter.title}

Research Context:
{context}

Write a comprehensive research summary for this chapter that:
1. Synthesizes key findings across all subjects
2. Identifies major themes and connections
3. Notes any contradictions or debates
4. Highlights the most important sources
5. Suggests areas that may need more research

Write in an academic but accessible style."""

        messages = [
            {"role": "system", "content": "You are a research synthesis expert."},
            {"role": "user", "content": prompt}
        ]

        try:
            return self.llm.chat(messages, temperature=0.4)
        except Exception as e:
            logger.error(f"LLM synthesis failed: {e}")
            return "\n\n".join(sr.synthesis for sr in chapter.subject_research if sr.synthesis)

    def execute(self) -> Dict[str, Any]:
        """Execute Phase 4: Synthesize research into summaries."""
        can_run, reason = self.can_execute()
        if not can_run:
            return {'success': False, 'error': reason}

        self.project.set_phase(4, 'in_progress')

        results = {
            'success': True,
            'chapters_synthesized': 0,
            'errors': [],
        }

        for i, chapter in enumerate(self.project.chapters):
            progress = i / len(self.project.chapters) if self.project.chapters else 0
            self.report_progress(f"Synthesizing: {chapter.title}", progress)

            if self.dry_run:
                logger.info(f"[DRY RUN] Would synthesize chapter: {chapter.title}")
                results['chapters_synthesized'] += 1
                continue

            try:
                chapter.update_status('synthesizing')
                chapter.research_summary = self.synthesize_chapter(chapter)
                chapter.update_status('complete')
                results['chapters_synthesized'] += 1
            except Exception as e:
                logger.error(f"Synthesis failed for chapter '{chapter.title}': {e}")
                results['errors'].append({
                    'chapter': chapter.chapter_number,
                    'error': str(e),
                })

        self.project.set_phase(4, 'completed')
        self.project.save()

        self.report_progress("Synthesis complete", 1.0)
        return results


# =============================================================================
# PHASE 5: DRAFT GENERATION
# =============================================================================

class Phase5Executor(PhaseExecutor):
    """Phase 5: Draft Generation - Create chapter drafts (optional)."""

    phase_number = 5
    phase_name = "Draft Generation"

    # Extended writing style presets
    WRITING_PRESETS = {
        'academic': """Write in formal academic style:
- Use precise, scholarly language
- Maintain objective tone
- Include appropriate hedging for uncertain claims
- Use topic sentences and clear paragraph structure
- Prefer active voice where appropriate""",

        'accessible': """Write in academic but accessible style:
- Explain technical terms when first used
- Use concrete examples to illustrate abstract concepts
- Vary sentence length for readability
- Avoid unnecessary jargon
- Maintain scholarly rigor while being approachable""",

        'narrative': """Write in engaging narrative style:
- Use storytelling techniques where appropriate
- Create flow between paragraphs
- Vary rhythm and pacing
- Make historical figures come alive
- Balance narrative with scholarly content""",

        'concise': """Write concisely:
- Eliminate redundancy
- Prefer shorter sentences
- Remove filler words
- One idea per paragraph
- Get to the point quickly""",

        'popular': """Write for general readers:
- Assume no prior knowledge of the subject
- Define all technical terms
- Use analogies and comparisons to familiar concepts
- Engage the reader's curiosity
- Break up dense content with examples""",

        'technical': """Write in technical documentation style:
- Use precise terminology consistently
- Define terms on first use
- Structure content hierarchically
- Include clear examples and explanations
- Avoid ambiguity""",

        'journalistic': """Write in journalistic feature style:
- Lead with the most compelling angle
- Use concrete details and anecdotes
- Keep paragraphs short
- Balance exposition with narrative
- Engage readers with human interest elements""",

        'literary': """Write in literary nonfiction style:
- Employ rich, evocative language
- Use metaphor and imagery thoughtfully
- Create atmosphere and mood
- Balance craft with clarity
- Let ideas breathe through elegant prose""",

        'textbook': """Write in textbook style:
- Organize content with clear learning objectives
- Use numbered lists and bullet points for key concepts
- Include summary sections
- Define key terms in bold
- Progress from simple to complex""",

        'conversational': """Write in conversational academic style:
- Use first/second person where appropriate
- Ask rhetorical questions to engage readers
- Include asides and digressions when illuminating
- Balance informality with substance
- Write as if explaining to a curious friend"""
    }

    def __init__(
        self,
        project: BookProject,
        checkpoint: WorkflowCheckpoint,
        evidence_first: bool = False,
        writing_style: Optional[str] = None,
        writing_preset: Optional[str] = None,
        writing_prompt_file: Optional[str] = None,
        reference_voice: Optional[str] = None,
        polish_drafts: bool = False,
        delegate_drafting: bool = False,
        **kwargs
    ):
        super().__init__(project, checkpoint, **kwargs)
        self.evidence_first = evidence_first
        self.polish_drafts = polish_drafts
        self.delegate_drafting = delegate_drafting

        # Resolve writing style
        self.style_prompt = self._resolve_writing_style(
            writing_style, writing_preset, writing_prompt_file, reference_voice
        )

        # Only initialize LLM if we're delegating drafting
        self.llm = None
        if self.delegate_drafting:
            try:
                from research_agent import LLMInterface
                self.llm = LLMInterface()
            except ImportError:
                logger.warning("LLMInterface not available for delegated drafting")

    def _resolve_writing_style(
        self,
        writing_style: Optional[str],
        writing_preset: Optional[str],
        writing_prompt_file: Optional[str],
        reference_voice: Optional[str]
    ) -> str:
        """Resolve the writing style from various input sources."""
        # Custom free-text style
        if writing_style:
            return f"WRITING STYLE:\n{writing_style}"

        # Preset style
        if writing_preset and writing_preset in self.WRITING_PRESETS:
            return f"WRITING STYLE:\n{self.WRITING_PRESETS[writing_preset]}"

        # Style from file
        if writing_prompt_file:
            try:
                with open(writing_prompt_file, 'r', encoding='utf-8') as f:
                    return f"WRITING STYLE:\n{f.read()}"
            except Exception as e:
                logger.warning(f"Could not read style prompt file: {e}")

        # Voice cloning from reference
        if reference_voice:
            try:
                from polish_draft import analyze_reference_style
                with open(reference_voice, 'r', encoding='utf-8') as f:
                    reference_text = f.read()
                return analyze_reference_style(reference_text)
            except Exception as e:
                logger.warning(f"Could not analyze reference voice: {e}")

        # Default: accessible academic
        return f"WRITING STYLE:\n{self.WRITING_PRESETS['accessible']}"

    def can_execute(self) -> tuple[bool, str]:
        if self.project.phase_status.get(4) != 'completed':
            return False, "Phase 4 not completed"
        if self.delegate_drafting and not self.llm:
            return False, "LLM not available for delegated draft generation"
        return True, "Ready"

    def _get_chapter_quotes(self, chapter: ChapterProject) -> List[Dict[str, Any]]:
        """Extract quotes binned to this chapter's subjects."""
        quotes = []
        for subject in chapter.subjects:
            for research in chapter.subject_research:
                if research.subject == subject:
                    for result in research.results:
                        quotes.append({
                            'text': result.get('chunk_text', '')[:500],
                            'source': result.get('document_id', ''),
                            'title': result.get('title', 'Unknown'),
                            'author': result.get('author', 'Unknown'),
                            'subject': subject
                        })
        return quotes[:30]  # Limit to top 30 quotes

    def generate_chapter_draft_evidence_first(self, chapter: ChapterProject) -> str:
        """
        Generate a draft using ONLY the binned quotes (evidence-first).
        This prevents hallucination by forcing the LLM to synthesize rather than generate.
        """
        if not self.llm:
            return f"# {chapter.title}\n\n{chapter.research_summary}"

        # Get quotes binned to this chapter
        quotes = self._get_chapter_quotes(chapter)

        if not quotes:
            logger.warning(f"No quotes found for chapter '{chapter.title}' - falling back to summary mode")
            return self.generate_chapter_draft(chapter)

        # Format quotes for the prompt
        quote_sections = []
        for i, q in enumerate(quotes, 1):
            quote_sections.append(
                f"[{i}] \"{q['text'][:300]}...\"\n"
                f"    — {q['author']}, *{q['title']}* [{q['source']}]"
            )
        quotes_text = "\n\n".join(quote_sections)

        prompt = f"""You are writing a book chapter using ONLY the provided source quotes.

Chapter Title: {chapter.title}
Subjects Covered: {', '.join(chapter.subjects)}

=== SOURCE QUOTES (Use ONLY these) ===
{quotes_text}
=== END SOURCES ===

{self.style_prompt}

CRITICAL RULES:
1. ONLY use information from the quotes above - do NOT add any facts not in the sources
2. Cite each quote using [N] format when you use it
3. If you cannot find information for a topic, write "[RESEARCH NEEDED: topic]" instead of making up content
4. Synthesize and connect the quotes - do not just list them

Write a well-structured chapter draft that:
1. Opens with an introduction based on the key themes in the sources
2. Organizes the quoted material into coherent sections
3. Uses transitions to connect ideas between quotes
4. Concludes by synthesizing the main insights

Apply the writing style guidelines above. Use [N] citations throughout."""

        messages = [
            {"role": "system", "content": "You are an expert academic writer who ONLY uses provided sources."},
            {"role": "user", "content": prompt}
        ]

        try:
            draft = self.llm.chat(messages, temperature=0.3)  # Lower temperature for evidence-first
            return f"# {chapter.title}\n\n{draft}"
        except Exception as e:
            logger.error(f"Evidence-first draft generation failed: {e}")
            return f"# {chapter.title}\n\n{chapter.research_summary}"

    def generate_chapter_draft(self, chapter: ChapterProject) -> str:
        """Generate a draft for a chapter (standard mode)."""
        if not self.llm:
            return f"# {chapter.title}\n\n{chapter.research_summary}"

        prompt = f"""You are writing a book chapter draft.

Chapter Title: {chapter.title}

Research Summary:
{chapter.research_summary}

Subjects Covered: {', '.join(chapter.subjects)}

{self.style_prompt}

Write a well-structured chapter draft that:
1. Opens with an engaging introduction to the chapter's themes
2. Covers each subject area systematically
3. Uses the research to support key points
4. Includes transitions between sections
5. Concludes with a summary connecting to broader themes

Apply the writing style guidelines above.
Include placeholder citations like [Author_Year] where sources should be cited."""

        messages = [
            {"role": "system", "content": "You are an expert academic writer."},
            {"role": "user", "content": prompt}
        ]

        try:
            draft = self.llm.chat(messages, temperature=0.5)
            return f"# {chapter.title}\n\n{draft}"
        except Exception as e:
            logger.error(f"Draft generation failed: {e}")
            return f"# {chapter.title}\n\n{chapter.research_summary}"

    def _polish_draft(self, draft_content: str) -> str:
        """Apply polishing pass to a draft using the configured style."""
        try:
            from polish_draft import polish_text
            result = polish_text(
                text=draft_content,
                style=self.style_prompt,
                preserve_citations=True,
                section_by_section=True
            )
            return result.polished_text
        except Exception as e:
            logger.warning(f"Polish pass failed, using original draft: {e}")
            return draft_content

    def generate_chapter_brief(self, chapter: ChapterProject) -> Dict[str, Any]:
        """
        Generate a chapter brief for Claude Code to write.

        This is the DEFAULT mode when --delegate-drafting is NOT specified.
        Returns research, outline, and context for Claude Code to write the draft.

        Args:
            chapter: The chapter to generate a brief for

        Returns:
            Dict containing research, outline, quotes, and style guidance
        """
        quotes = self._get_chapter_quotes(chapter)

        # Build structured brief
        return {
            'chapter_number': chapter.chapter_number,
            'title': chapter.title,
            'subjects': [s.topic if hasattr(s, 'topic') else str(s) for s in chapter.subjects],

            'research_summary': chapter.research_summary or "No research summary available.",

            'quotes': [
                {
                    'text': q['text'],
                    'source': q['source'],
                    'author': q.get('author', 'Unknown'),
                    'title': q.get('title', 'Unknown'),
                    'subject': q.get('subject', '')
                }
                for q in quotes
            ],

            'sources_used': [
                {
                    'document_id': s.get('document_id'),
                    'title': s.get('title', 'Unknown'),
                    'author': s.get('author', 'Unknown')
                }
                for s in chapter.sources_used
            ],

            'gaps_identified': [
                {'query': g.get('query'), 'status': g.get('status', 'pending')}
                for g in chapter.gaps_identified
            ],

            'writing_style': self.style_prompt,

            'instructions': (
                "Claude Code should use this brief to write the chapter draft. "
                "Use the research summary as the foundation, incorporate quotes with citations, "
                "and follow the writing style guidelines. Mark any information gaps with "
                "[RESEARCH NEEDED: topic] placeholders."
            ),
        }

    def execute(self) -> Dict[str, Any]:
        """Execute Phase 5: Generate chapter drafts or briefs.

        If delegate_drafting is True: Uses internal LLM to generate full drafts.
        If delegate_drafting is False (default): Returns chapter briefs for Claude Code.
        """
        can_run, reason = self.can_execute()
        if not can_run:
            return {'success': False, 'error': reason}

        self.project.set_phase(5, 'in_progress')

        # Determine mode based on delegate_drafting flag
        if self.delegate_drafting:
            return self._execute_delegated_drafting()
        else:
            return self._execute_brief_generation()

    def _execute_delegated_drafting(self) -> Dict[str, Any]:
        """Execute delegated drafting mode - use internal LLM to generate drafts."""
        mode = "evidence-first" if self.evidence_first else "standard"
        logger.info(f"DELEGATED MODE: Draft generation using internal LLM ({mode})")
        if self.style_prompt:
            logger.info(f"Writing style configured")
        if self.polish_drafts:
            logger.info(f"Polish pass enabled")

        results = {
            'success': True,
            'mode': 'delegated',
            'generation_mode': mode,
            'drafts_generated': 0,
            'drafts_polished': 0,
            'errors': [],
        }

        for i, chapter in enumerate(self.project.chapters):
            progress = i / len(self.project.chapters) if self.project.chapters else 0
            self.report_progress(f"Drafting ({mode}): {chapter.title}", progress)

            if self.dry_run:
                logger.info(f"[DRY RUN] Would draft chapter: {chapter.title}")
                results['drafts_generated'] += 1
                continue

            try:
                chapter.update_status('drafting')
                if self.evidence_first:
                    chapter.draft_content = self.generate_chapter_draft_evidence_first(chapter)
                else:
                    chapter.draft_content = self.generate_chapter_draft(chapter)

                # Apply polish pass if requested
                if self.polish_drafts and chapter.draft_content:
                    self.report_progress(f"Polishing: {chapter.title}", progress + 0.05)
                    chapter.draft_content = self._polish_draft(chapter.draft_content)
                    results['drafts_polished'] += 1

                chapter.update_status('complete')
                results['drafts_generated'] += 1
            except Exception as e:
                logger.error(f"Draft generation failed for chapter '{chapter.title}': {e}")
                results['errors'].append({
                    'chapter': chapter.chapter_number,
                    'error': str(e),
                })

        self.project.set_phase(5, 'completed')
        self.project.completed_at = datetime.now().isoformat()
        self.project.save()

        self.report_progress("Draft generation complete", 1.0)
        return results

    def _execute_brief_generation(self) -> Dict[str, Any]:
        """Execute brief generation mode - return chapter briefs for Claude Code to write."""
        logger.info("DEFAULT MODE: Generating chapter briefs for Claude Code")

        results = {
            'success': True,
            'mode': 'briefs',
            'chapter_briefs': [],
            'note': 'Claude Code should write drafts using these briefs',
        }

        for i, chapter in enumerate(self.project.chapters):
            progress = i / len(self.project.chapters) if self.project.chapters else 0
            self.report_progress(f"Generating brief: {chapter.title}", progress)

            if self.dry_run:
                logger.info(f"[DRY RUN] Would generate brief for chapter: {chapter.title}")
                results['chapter_briefs'].append({
                    'chapter_number': chapter.chapter_number,
                    'title': chapter.title,
                    'dry_run': True
                })
                continue

            try:
                brief = self.generate_chapter_brief(chapter)
                results['chapter_briefs'].append(brief)
                chapter.update_status('brief_ready')
            except Exception as e:
                logger.error(f"Brief generation failed for chapter '{chapter.title}': {e}")
                results['chapter_briefs'].append({
                    'chapter_number': chapter.chapter_number,
                    'title': chapter.title,
                    'error': str(e)
                })

        # Mark phase as ready for drafting (not completed - Claude Code still needs to write)
        self.project.set_phase(5, 'briefs_ready')
        self.project.save()

        self.report_progress("Chapter briefs generated - ready for Claude Code to write drafts", 1.0)
        return results


# =============================================================================
# WORKFLOW RUNNER
# =============================================================================

def get_phase_executor(
    phase: int,
    project: BookProject,
    checkpoint: WorkflowCheckpoint,
    **kwargs
) -> PhaseExecutor:
    """Get the appropriate phase executor."""
    executors = {
        1: Phase1Executor,
        2: Phase2Executor,
        3: Phase3Executor,
        4: Phase4Executor,
        5: Phase5Executor,
    }

    executor_class = executors.get(phase)
    if not executor_class:
        raise ValueError(f"Invalid phase: {phase}")

    return executor_class(project, checkpoint, **kwargs)


def run_workflow(
    project: BookProject,
    start_phase: int = 1,
    end_phase: int = 5,
    skip_phases: List[int] = None,
    progress_callback: Optional[Callable[[str, float], None]] = None,
    gap_threshold: float = 0.0,
    evidence_first: bool = False,
    writing_style: Optional[str] = None,
    writing_preset: Optional[str] = None,
    writing_prompt_file: Optional[str] = None,
    reference_voice: Optional[str] = None,
    polish_drafts: bool = False,
    delegate_drafting: bool = False,
    **kwargs
) -> Dict[str, Any]:
    """
    Run the complete workflow or a portion of it.

    Args:
        project: BookProject to process
        start_phase: Phase to start from (1-5)
        end_phase: Phase to end at (1-5)
        skip_phases: Phases to skip
        progress_callback: Progress callback function
        gap_threshold: Minimum research coverage % before drafting (0-100)
        evidence_first: Use evidence-first generation (quote-binning)
        writing_style: Custom writing style description
        writing_preset: Predefined writing style preset name
        writing_prompt_file: Path to file with custom style prompt
        reference_voice: Path to text file for voice cloning
        polish_drafts: Apply polishing pass after draft generation
        delegate_drafting: If True, use internal LLM to generate drafts.
                          If False (default), return chapter briefs for Claude Code.
        **kwargs: Additional arguments for phase executors

    Returns:
        Results dictionary with phase results
    """
    skip_phases = skip_phases or []

    # Load or create checkpoint
    project_dir = Path(project.project_dir)
    checkpoint = WorkflowCheckpoint.load(project_dir)
    if checkpoint is None:
        checkpoint = WorkflowCheckpoint(
            book_project_id=project.project_id,
            phase=start_phase,
            chapter_index=0,
            subject_index=0,
        )
        checkpoint.compute_hash(project)

    results = {
        'project_id': project.project_id,
        'phases': {},
        'success': True,
    }

    for phase in range(start_phase, end_phase + 1):
        if phase in skip_phases:
            logger.info(f"Skipping Phase {phase}")
            continue

        # Gap threshold check before Phase 5 (Draft Generation)
        if phase == 5 and gap_threshold > 0:
            gap_coverage = _calculate_gap_coverage(project)
            if gap_coverage < gap_threshold:
                logger.warning(f"Gap coverage {gap_coverage:.0f}% below threshold {gap_threshold:.0f}%")
                results['halted_for_gaps'] = True
                results['gap_coverage'] = gap_coverage
                results['pending_research_queries'] = _get_pending_research_queries(project)
                results['success'] = False
                return results

        logger.info(f"Starting Phase {phase}")
        checkpoint.set_phase(phase)
        checkpoint.save(project_dir)

        try:
            # Pass writing style options only to Phase 5
            phase_kwargs = dict(kwargs)
            if phase == 5:
                phase_kwargs.update({
                    'writing_style': writing_style,
                    'writing_preset': writing_preset,
                    'writing_prompt_file': writing_prompt_file,
                    'reference_voice': reference_voice,
                    'polish_drafts': polish_drafts,
                    'delegate_drafting': delegate_drafting,
                })

            executor = get_phase_executor(
                phase, project, checkpoint,
                progress_callback=progress_callback,
                evidence_first=evidence_first,
                **phase_kwargs
            )

            can_run, reason = executor.can_execute()
            if not can_run:
                logger.warning(f"Cannot run Phase {phase}: {reason}")
                results['phases'][phase] = {'skipped': True, 'reason': reason}
                continue

            phase_result = executor.execute()
            results['phases'][phase] = phase_result

            if not phase_result.get('success', False):
                results['success'] = False
                break

        except Exception as e:
            logger.exception(f"Phase {phase} failed with error")
            results['phases'][phase] = {'success': False, 'error': str(e)}
            results['success'] = False
            break

    return results


def _calculate_gap_coverage(project: BookProject) -> float:
    """Calculate research coverage percentage (100 - gap percentage)."""
    total_gaps = len(project.all_gaps) if hasattr(project, 'all_gaps') else 0
    filled_gaps = sum(1 for g in project.all_gaps if g.get('filled', False)) if hasattr(project, 'all_gaps') else 0

    if total_gaps == 0:
        return 100.0  # No gaps = full coverage

    return ((total_gaps - (total_gaps - filled_gaps)) / total_gaps) * 100


def _get_pending_research_queries(project: BookProject) -> List[str]:
    """Get list of research queries needed to fill gaps."""
    queries = []
    if hasattr(project, 'all_gaps'):
        for gap in project.all_gaps:
            if not gap.get('filled', False):
                query = gap.get('suggested_query') or gap.get('description', '')
                if query:
                    queries.append(query)
    return queries[:10]  # Return top 10


# =============================================================================
# EXPORTS
# =============================================================================

__all__ = [
    'PhaseExecutor',
    'Phase1Executor',
    'Phase2Executor',
    'Phase3Executor',
    'Phase4Executor',
    'Phase5Executor',
    'get_phase_executor',
    'run_workflow',
]
