# Writing Project Workflow Guide

A comprehensive step-by-step guide for working with Claude Code to create books, essays, and research projects using the Research Development Framework.

---

## Overview

This guide bridges **CLI tooling** (hard logic, storage, data integrity) with the **LLM Agent** (soft logic, generation, synthesis). Each phase is designed to prevent common failure modes: context overflow, hallucination, data corruption, and workflow drift.

```
┌─────────────────────────────────────────────────────────────────────────────┐
│                    COMPLETE WRITING PROJECT WORKFLOW                         │
├─────────────────────────────────────────────────────────────────────────────┤
│                                                                              │
│  1. INGEST     2. SCOPE      3. RESEARCH    4. GAP         5. SYNTHESIZE    │
│  ┌────────┐   ┌────────┐    ┌────────┐    ┌────────┐     ┌────────┐        │
│  │Add docs│ → │Outline │ →  │Search &│ →  │Analyze │  →  │Chapter │        │
│  │to lib  │   │& Refine│    │organize│    │& fill  │     │briefs  │        │
│  └────────┘   └────────┘    └────────┘    └────────┘     └────────┘        │
│                                                                │             │
│                                                                ▼             │
│  9. COMPILE    8. POLISH    7.5 HANDSHAKE  7. REVISE     6. DRAFT           │
│  ┌────────┐   ┌────────┐    ┌────────┐    ┌────────┐    ┌────────┐         │
│  │Editor- │ ← │Final   │ ←  │Chapter │ ←  │Edit &  │ ←  │Write   │         │
│  │in-Chief│   │refine  │    │links   │    │flow    │    │chapters│         │
│  └────────┘   └────────┘    └────────┘    └────────┘    └────────┘         │
│                                                                              │
└─────────────────────────────────────────────────────────────────────────────┘

LINKED CHAPTER FLOW (Phases 5-6):
┌────────┐     ┌────────┐     ┌────────┐     ┌────────┐
│ Ch1    │────→│ Ch2    │────→│ Ch3    │────→│ Ch4    │  ...
│ Brief  │     │ Brief  │     │ Brief  │     │ Brief  │
└────────┘     └────────┘     └────────┘     └────────┘
     │              │              │              │
     ▼              ▼              ▼              ▼
┌────────┐     ┌────────┐     ┌────────┐     ┌────────┐
│ Ch1    │────→│ Ch2    │────→│ Ch3    │────→│ Ch4    │  ...
│ Draft  │     │ Draft  │     │ Draft  │     │ Draft  │
└────────┘     └────────┘     └────────┘     └────────┘
           ↑              ↑              ↑
     "Baton Pass": Each chapter receives the ending of its predecessor
```

**Key Principles:**
- **CLI for Data Integrity**: Bibliography generation, citation verification, and search are CLI commands—never LLM prompts
- **Synthesis Before Drafting**: Never ask the LLM to draft from raw research files
- **Outline Before Research**: Finalize structure before deep research to avoid reorganization
- **Chapters Are Linked**: Each chapter must connect to its neighbors—this is a book, not 8 essays
- **Git for Safety**: Commit after every major phase

---

## Table of Contents

1. [Phase 1: Document Ingestion](#phase-1-document-ingestion)
2. [Phase 2: Scoping & Structure](#phase-2-scoping--structure)
3. [Phase 3: Research Assembly](#phase-3-research-assembly)
4. [Phase 4: Gap Analysis](#phase-4-gap-analysis)
5. [Phase 5: Synthesis](#phase-5-synthesis) ← Includes "Bridge Strategy"
6. [Phase 6: Drafting](#phase-6-drafting) ← Includes Context Injection
7. [Phase 7: Revision](#phase-7-revision)
8. [Phase 7.5: Handshake Check](#phase-75-handshake-check) ← Chapter connections
9. [Phase 8: Polish](#phase-8-polish)
10. [Phase 9: Compilation](#phase-9-compilation) ← Includes Editor-in-Chief

---

## Session Resumption

When resuming work on an existing project, Claude Code can quickly restore context using the `rdf context` command:

```bash
./rdf context BOOK_xxx --format json
```

This returns a comprehensive context packet including:
- Current phase and progress
- Active chapter and subject
- Recent activity log
- Pending decisions and blocking items
- Resource status (API credits, budget)

**Example workflow resumption:**

```bash
# 1. Get context packet for warm-start
./rdf context BOOK_xxx

# 2. Check queue for any pending approvals
./rdf queue list --project BOOK_xxx

# 3. Continue workflow from current phase
./rdf book projects/books/BOOK_xxx/outline.yaml --resume
```

---

## Project Structure

Before starting, understand the complete project structure:

```
projects/books/BOOK_xxx/
├── outline.yaml              ← Project configuration (finalized in Phase 2)
├── workflow_state.json       ← Tracks current phase
├── scratchpad.md             ← Cross-chapter notes (critical for revision)
│
├── research/                 ← Phase 3: Raw research by chapter
│   ├── ch01_research.md
│   ├── ch02_research.md
│   ├── quotes_collection.md
│   └── web_research/         ← Tavily results (if used)
│
├── briefs/                   ← Phase 5: Synthesized chapter briefs
│   ├── ch01_brief.md         ← Structured outline with mapped quotes
│   ├── ch02_brief.md
│   └── ...
│
├── drafts/                   ← Phase 6: First drafts
│   ├── chapter_01_draft.md
│   └── ...
│
├── revised/                  ← Phase 7: After editing pass
│   ├── chapter_01_revised.md
│   └── ...
│
├── polished/                 ← Phase 8: Final chapter versions
│   ├── chapter_01.md
│   └── ...
│
└── compiled/                 ← Phase 9: Final manuscript
    ├── bibliography.md       ← CLI-generated (not LLM)
    ├── manuscript.md
    └── final/
```

---

## Phase 1: Document Ingestion

### Purpose
Add your source documents to the searchable library.

### Step 1.1: Place Documents

Copy research documents to the import folder:

```bash
cp ~/my_research/*.pdf NEW_DOCS/
cp ~/my_research/*.docx NEW_DOCS/
```

**Supported formats:** PDF, DOCX, DOC, ODT, RTF, TXT, MD, HTML

**Note:** The CLI handles filename collisions by appending timestamps. Duplicate content is detected during ingestion.

### Step 1.2: Ingest Documents

**Tell Claude:**

> "Ingest the documents I placed in NEW_DOCS."

**What Claude will do:**

```bash
./rdf ingest NEW_DOCS/
```

### Step 1.3: Verify Ingestion

**Tell Claude:**

> "Check the library health and show me ingestion results."

**What Claude will do:**

```bash
./rdf health --format json
```

### Step 1.4: Assess Quality

For documents flagged as low quality:

> "Assess document DOC_xxx and tell me if it needs re-OCR."

```bash
./rdf assess DOC_xxx --format json
```

### Git Checkpoint

```bash
# No project files yet, but if you modified library:
git add library/
git commit -m "Phase 1: Ingested new research documents"
```

---

## Phase 2: Scoping & Structure

### Purpose
Define and FINALIZE the project structure BEFORE deep research. This prevents the "outline paradox" where research organization becomes obsolete after outline changes.

### Step 2.1: Create Initial Outline

**Tell Claude:**

> "Create a new book project about 'History of Freemasonry in America' with approximately 8 chapters targeting 200 pages."

**What Claude will do:**

```bash
./rdf outline "History of Freemasonry in America" --chapters 8 --pages 200
```

This creates:
```
projects/books/BOOK_[timestamp]/
├── outline.yaml           ← Initial outline
├── workflow_state.json
└── scratchpad.md          ← Empty, for cross-chapter notes
```

### Step 2.2: Review and Refine Outline (CRITICAL)

**Do this BEFORE any research organization.**

**Tell Claude:**

> "Show me the outline for this project. I want to review and potentially modify the chapter structure."

Review the `outline.yaml`. Make all structural changes NOW:

> "Update the outline:
> - Split Chapter 3 into two chapters: 'Early Lodge Formation' and 'Colonial Adaptation'
> - Add a new chapter on 'Masonic Architecture in America'
> - Target 250 pages total
> - Reorder chapters chronologically"

### Step 2.3: Set Project Parameters

**Tell Claude:**

> "Set these project parameters:
> - Style: accessible historical narrative
> - Audience: educated general readers
> - Citation style: Chicago Notes & Bibliography
> - Autonomy level: supervised"

### Step 2.4: Define Chapter Details

For each chapter, specify focus and key sources:

> "For Chapter 1 'Colonial Origins':
> - Target pages: 25-30
> - Key themes: English lodge traditions, colonial adaptation, early membership
> - Known relevant documents: DOC_012, DOC_045
> - Tone: narrative, scene-setting"

### Final Outline Structure

```yaml
# outline.yaml
title: "History of Freemasonry in America"
author: "[Your Name]"
target_pages: 250
style: accessible
audience: educated_general
citation_style: chicago

chapters:
  - number: 1
    title: "Colonial Origins"
    target_pages: 30
    key_themes:
      - English lodge traditions
      - Colonial adaptation
    focus: "Setting the scene for American Freemasonry"

  - number: 2
    title: "The Founding Generation"
    target_pages: 35
    key_themes:
      - Washington and the lodges
      - Revolutionary symbolism
    focus: "Masonic influence on the founders"
```

### Git Checkpoint

```bash
git add projects/books/BOOK_xxx/
git commit -m "Phase 2: Finalized outline and project structure"
```

---

## Phase 3: Research Assembly

### Purpose
Gather and organize research materials according to the FINALIZED outline.

### Step 3.1: Conduct Chapter Research

**Tell Claude:**

> "Research Chapter 1 topics using the library. Search for:
> - Early American Masonic lodges (1730-1770)
> - English Grand Lodge influence on colonies
> - First lodge charters in Boston, Philadelphia
>
> Save results to the project's research folder."

**What Claude will do:**

```bash
./rdf research "Early American Masonic lodges colonial period" --strict \
  -o projects/books/BOOK_xxx/research/ch01_research.md

./rdf research "English Grand Lodge colonial America" --strict \
  -o projects/books/BOOK_xxx/research/ch01_english_influence.md
```

### Step 3.2: Extract Quotes

**Tell Claude:**

> "Extract the 30 best quotes about George Washington and Freemasonry with full citations."

**What Claude will do:**

```bash
./rdf quotes "George Washington Freemasonry" --max 30 --format json \
  > projects/books/BOOK_xxx/research/ch02_washington_quotes.json
```

### Step 3.3: Organize Research by Chapter

Now that the outline is finalized, organization won't become obsolete:

> "Organize all gathered research by chapter according to the outline. Create one research file per chapter."

**Structure after research:**

```
projects/books/BOOK_xxx/
├── outline.yaml
├── research/
│   ├── ch01_research.md      ← All Ch1 materials
│   ├── ch02_research.md      ← All Ch2 materials
│   ├── ch03_research.md
│   ├── quotes_by_chapter.md  ← Organized quote collection
│   └── source_index.md       ← Quick reference to all sources used
```

### Git Checkpoint

```bash
git add projects/books/BOOK_xxx/research/
git commit -m "Phase 3: Research assembly complete"
```

---

## Phase 4: Gap Analysis

### Purpose
Identify missing research BEFORE attempting synthesis or drafting.

### Step 4.1: Analyze Research Coverage

**Tell Claude:**

> "Analyze the research gathered for each chapter against the outline requirements. Identify gaps where I don't have sufficient source material."

Claude should produce a gap report:

```markdown
# Research Gap Analysis

## Chapter 3: Masonic Architecture
### Gaps Identified:
1. **Capitol Building symbolism** - No sources on Masonic influence in DC design
2. **Lodge building traditions** - Only 2 sources, need primary materials

### Research Needed:
- Documents on L'Enfant's Masonic connections
- Primary sources on specific lodge buildings

## Chapter 5: Civil War Era
### Gaps Identified:
1. **Confederate lodges** - No coverage of Southern Masonry during the war
```

### Step 4.2: Gap Resolution Strategy

Choose how to handle gaps:

**Option A: Require Complete Research (Strict)**

> "Generate a list of documents I need to obtain. Do not proceed until gaps are filled."

**Option B: Allow Limited Synthesis**

> "For minor gaps, synthesize from related sources with explicit hedging. Flag major gaps for additional research."

**Option C: Use Web Research (Tavily)**

> "Use Tavily to research the Capitol Building symbolism gap. Show me results for approval before incorporating."

### Step 4.3: Tavily Web Research (If Used)

**Tell Claude:**

> "Search the web for 'Masonic symbolism Washington DC Capitol L'Enfant'. Save results for my review."

```bash
./rdf research "Masonic symbolism Washington DC Capitol L'Enfant" --allow-web \
  -o projects/books/BOOK_xxx/research/web_research/ch03_capitol.md
```

**Review before approval:**

> "Show me the Tavily results. I want to verify sources before approving them."

**Approve or reject:**

> "Approve sources 1, 3, and 5. Reject source 2 (unreliable blog). I'll find a replacement for source 4."

### Step 4.4: Ingest Additional Documents

If you obtain new documents:

> "I've added three new PDFs about Masonic architecture to NEW_DOCS. Ingest them and re-analyze gaps for Chapter 3."

### Git Checkpoint

```bash
git add projects/books/BOOK_xxx/
git commit -m "Phase 4: Gap analysis complete, research sufficient"
```

---

## Phase 5: Synthesis

### Purpose
Transform raw research into structured **Chapter Briefs** that the LLM can draft from without context overflow.

**Why This Phase Exists:**
- Raw research files may be 50+ pages of excerpts
- Feeding raw research directly causes "lost in the middle" degradation
- Synthesis creates a structured map from outline points to specific evidence

### Step 5.1: Create Chapter Brief

**Tell Claude:**

> "Using the `research_synthesis` prompt, create a Chapter Brief for Chapter 1.
>
> Transform the raw research in `research/ch01_research.md` into a structured brief that:
> - Maps each outline point to specific evidence
> - Identifies which quotes support which arguments
> - Notes areas where evidence is thin
> - Suggests paragraph-level structure"

**See:** [prompts/synthesis/research_synthesis.md](../prompts/synthesis/research_synthesis.md)

### Step 5.2: The "Bridge Strategy" (For Chapters 2+)

**Crucial:** To ensure the book flows as a single narrative, the Brief for Chapter N must know how Chapter N-1 ends. You cannot write Chapter 2 successfully without knowing exactly how Chapter 1 ended.

**Tell Claude (for Chapter 2 onwards):**

> "Update the Brief for Chapter 2. Add a 'Transition Strategy' section at the top.
>
> **Context:** Chapter 1 ends with [paste summary or last paragraph of Ch1].
> **Requirement:** The opening of Chapter 2 must pick up this thread.
> **Instruction:** Ensure the first section of the brief bridges from [Ch1's ending concept] to [Ch2's opening concept]."

**Brief Header for Ch2+ should include:**

```markdown
# Chapter 2 Brief: The Founding Generation

## Transition Strategy
**Previous chapter ends with:** [Summary of Ch1 ending - themes, unresolved questions]
**This chapter opens by:** [How we pick up the thread]
**Bridge concept:** [The linking idea between chapters]

## Target: 30-35 pages
...
```

This "Baton Pass" ensures each chapter receives the narrative thread from its predecessor.

```
LINKED BRIEFING FLOW:
┌──────────────┐       ┌──────────────┐       ┌──────────────┐
│  CH1 BRIEF   │       │  CH2 BRIEF   │       │  CH3 BRIEF   │
│              │       │              │       │              │
│  [Ending] ───┼──────→│ [Transition] │       │              │
│              │       │  [Ending] ───┼──────→│ [Transition] │
└──────────────┘       └──────────────┘       └──────────────┘
```

### Step 5.3: Chapter Brief Structure

The brief should look like:

```markdown
# Chapter 1 Brief: Colonial Origins

## Target: 25-30 pages

## Opening (2-3 paragraphs)
**Hook:** Use the 1733 Boston lodge charter (DOC_012, p.34)
**Context:** Establish English Grand Lodge backdrop

## Section 1: English Precedents (5-6 pages)
### Key Arguments:
1. Grand Lodge formation (1717) created exportable model
   - Evidence: DOC_045 pp.12-18 (direct description)
   - Quote: "The lodges of the colonies..." [use in paragraph 2]

2. Colonial merchants as transmission vectors
   - Evidence: DOC_078 membership rolls
   - Gap: Specific ship records weak, hedge with "likely"

### Paragraph Map:
- Para 1: Grand Lodge founding context
- Para 2: Constitutional framework [use quote from DOC_045]
- Para 3-4: Merchant networks
- Para 5: First colonial charters

## Section 2: Boston Emergence (4-5 pages)
[...]

## Thin Areas (Address in Draft)
- Limited primary sources for 1730s membership
- Economic motivations speculative

## Suggested Chapter Flow
Opening → English context → Boston → Philadelphia → Synthesis
```

### Step 5.3: Review and Approve Briefs

**Tell Claude:**

> "Show me the Chapter 1 brief. I want to verify the evidence mapping before drafting."

Make adjustments:

> "In the brief, move the Washington letter to Section 2 instead of the opening. Add a note to address the counter-argument about exaggerated Masonic influence."

### Output Structure

```
projects/books/BOOK_xxx/
├── briefs/
│   ├── ch01_brief.md         ← Reviewed and approved
│   ├── ch02_brief.md
│   └── ...
```

### Git Checkpoint

```bash
git add projects/books/BOOK_xxx/briefs/
git commit -m "Phase 5: Chapter briefs synthesized and approved"
```

---

## Phase 6: Drafting

### Purpose
Write initial chapter drafts using **Chapter Briefs**, not raw research.

### Step 6.1: Select Drafting Prompt

| Content Type | Prompt | Best For |
|--------------|--------|----------|
| Historical | `scholarly_narrative` | Evidence-based narrative |
| Philosophical | `living_thought` | Contemplative, esoteric |
| Academic | `essay_academic` | Structured argumentation |

**See:** [prompts/drafting/](../prompts/drafting/)

### Citation Syntax (Critical)

**All drafts must use standard Markdown footnote syntax:**

```markdown
The Grand Lodge was established in 1717.[^1] This marked a pivotal moment
in Masonic history, transforming local traditions into a formal institution.[^2]

[^1]: Anderson, *Constitutions*, 34.
[^2]: Stevenson, *Origins of Freemasonry*, 112-115.
```

**Rules:**
- Use `[^1]`, `[^2]`, etc. for citation markers (NOT `<sup>` tags)
- Place reference definitions at the **bottom of each chapter file**
- Do NOT use parenthetical citations like `(Smith 2023)`
- Number citations sequentially within each chapter (restart at `[^1]` per chapter)

**Why this matters:** Markdown `[^n]` syntax is converted to superscript numbers during compilation. Whether they appear as footnotes (bottom of page) or endnotes (back of book) is determined by the **compilation settings**, not the draft.

### Step 6.2: Draft from Brief (with Context Injection)

**For Chapter 1:**

> "Draft Chapter 1 using the `scholarly_narrative` prompt.
>
> **Input:** Use the Chapter Brief in `briefs/ch01_brief.md`
> **Style:** accessible historical narrative
> **Audience:** educated general readers
> **Citations:** Chicago footnotes
> **Target:** 25-30 pages
>
> Follow the paragraph map in the brief. Include all mapped quotes with proper citations."

**For Chapters 2+ (Context Injection Required):**

**Tell Claude:**

> "Draft Chapter 2 using the `scholarly_narrative` prompt.
>
> **Input:** Use the Chapter Brief in `briefs/ch02_brief.md`
> **Context:** Here is the last page of Chapter 1:
> ```
> [Paste the final ~500 words of Chapter 1]
> ```
> **Instruction:** Ensure the opening of Chapter 2 connects smoothly to this text. The reader should feel a continuous narrative, not a new essay.
>
> **Style:** accessible historical narrative
> **Audience:** educated general readers
> **Citations:** Chicago footnotes
> **Target:** 30-35 pages"

**The "Baton Pass" in Action:**

```
CONTEXT INJECTION FLOW:
┌──────────────┐       ┌──────────────┐
│  CHAPTER 1   │       │  CHAPTER 2   │
│ (Completed)  │       │  (Drafting)  │
│              │       │              │
│ [Last Page] ─┼──────→│  [Opening]   │  ← The drafter sees how Ch1 ended
└──────────────┘       └──────────────┘
```

**Critical:** The LLM works from the brief (structured, ~5 pages) not raw research (~50 pages). For Ch2+, it also receives the ending of the previous chapter to maintain narrative continuity.

### Step 6.3: Review Draft

**Tell Claude:**

> "Show me the draft of Chapter 1. Highlight:
> - Any places where you couldn't find evidence in the brief
> - Any deviations from the paragraph map
> - Any thin areas that need flagging"

### Step 6.4: Request Draft Changes

If changes needed:

> "In Chapter 1:
> - Expand the section on early lodge meetings (add 2-3 paragraphs)
> - Remove the speculation about Franklin's motivations (no source in brief)
> - The opening is too dry—make it more engaging while keeping the 1733 charter"

### Step 6.5: Update Scratchpad

During drafting, note cross-chapter issues:

> "Add to the project scratchpad: 'Check date consistency of Philadelphia Lodge founding between Ch1 and Ch3—currently Ch1 says 1731, verify against Ch3 draft.'"

### Output Structure

```
projects/books/BOOK_xxx/
├── drafts/
│   ├── chapter_01_draft.md
│   ├── chapter_02_draft.md
│   └── ...
├── scratchpad.md             ← Updated with cross-chapter notes
```

### Git Checkpoint

```bash
git add projects/books/BOOK_xxx/drafts/ projects/books/BOOK_xxx/scratchpad.md
git commit -m "Phase 6: Chapter 1 drafted"
```

---

## Phase 7: Revision

### Purpose
Edit completed drafts for clarity, accuracy, and consistency. Address cross-chapter issues from the scratchpad.

### Step 7.1: Pre-Revision Checks

**Tell Claude:**

> "Before revising Chapter 1:
> 1. Show me the current scratchpad notes
> 2. List any issues from drafting that need resolution
> 3. Verify all citations are complete (no [verify] markers remaining)"

### Step 7.2: Select Revision Prompt

| Content Type | Prompt | Focus |
|--------------|--------|-------|
| Historical | `history_editor` | Narrative flow, footnote accuracy |
| General | `general_editor` | Structure, argument, clarity |

**See:** [prompts/revision/](../prompts/revision/)

### Step 7.3: Revise Chapter

**Tell Claude:**

> "Revise Chapter 1 using the `history_editor` prompt.
>
> **Focus on:**
> - Improving narrative flow
> - Removing clichés and redundancy
> - Strengthening transitions
> - Ensuring footnotes do real work
>
> **Preserve:** All factual claims and citations exactly as written
> **Address:** Scratchpad note about Philadelphia Lodge date"

### Step 7.4: Compare Draft to Revision

**Tell Claude:**

> "Show me the significant changes between draft and revision for Chapter 1."

**Note on Diff Tools:** Standard line-based diff is poor for prose. The `./rdf diff` command uses word-level comparison optimized for markdown, highlighting insertions and deletions within paragraphs.

```bash
./rdf diff drafts/chapter_01_draft.md revised/chapter_01_revised.md
```

### Step 7.5: Update Scratchpad

Mark resolved issues:

> "Update scratchpad: Mark the Philadelphia Lodge date issue as resolved (confirmed 1731 is correct per DOC_078)."

Add new issues discovered:

> "Add to scratchpad: 'Ch1 mentions "Grand Master of Pennsylvania" - verify this title was used in colonial period for Ch4 reference.'"

### Output Structure

```
projects/books/BOOK_xxx/
├── revised/
│   ├── chapter_01_revised.md
│   ├── chapter_02_revised.md
│   └── ...
├── scratchpad.md             ← Issues resolved/added
```

### Git Checkpoint

```bash
git add projects/books/BOOK_xxx/revised/ projects/books/BOOK_xxx/scratchpad.md
git commit -m "Phase 7: Chapter 1 revised"
```

---

## Phase 7.5: Handshake Check

### Purpose
Specifically review the **connection points between chapters**. This phase ensures that the book reads as a continuous narrative, not a series of isolated essays.

Phase 7 (Revision) checks for flow *within* a chapter. Phase 7.5 checks the **Entry and Exit points** between chapters.

### When to Run
After both Chapter N and Chapter N+1 are revised, run the Handshake Check on their connection.

### Step 7.5.1: The Handshake Check

**Tell Claude:**

> "Perform a 'Handshake Check' between Chapter 1 and Chapter 2.
>
> **Read:** The last 2 pages of Chapter 1 and the first 2 pages of Chapter 2.
>
> **Analyze:**
> 1. Does the transition feel jarring or smooth?
> 2. Is there repetitive contextualization (e.g., re-introducing a concept fully explained in Ch1)?
> 3. Does the tone shift abruptly?
> 4. Does Ch2's opening pick up a thread from Ch1's ending?
>
> **Suggest:** Specific edits to the Opening of Ch2 (and/or Ending of Ch1) to improve the flow."

### Step 7.5.2: Hooks & Hangers Inspection

For each chapter, verify:

**The Hook (Opening):**
- Does the opening paragraph grab the reader?
- Does it pick up the narrative thread from the previous chapter?
- Does it feel like a continuation, not a restart?

**The Hanger (Ending):**
- Does the ending leave a question or theme open that the next chapter resolves?
- Is there a forward pull, or does it feel conclusively "closed"?

**Tell Claude:**

> "Review the Hook and Hanger for Chapter 3:
>
> **Hook Check:** Does the opening connect to Chapter 2's ending? Does it feel like we're continuing a story?
>
> **Hanger Check:** Does the ending create anticipation for Chapter 4? What unresolved thread will carry the reader forward?
>
> **Suggest improvements** if either feels weak."

### Step 7.5.3: Connection Audit (Full Manuscript)

After all chapters are revised, run a complete connection audit:

**Tell Claude:**

> "Perform a Connection Audit across all chapter transitions:
>
> For each transition (Ch1→Ch2, Ch2→Ch3, etc.):
> - Rate the smoothness (1-5)
> - Identify the 'bridge concept' that links them
> - Flag any jarring transitions
>
> Create a report summarizing connection quality."

### Handshake Visualization

```
THE HANDSHAKE CHECK:
┌──────────────┐   |   ┌──────────────┐
│  CHAPTER 1   │   |   │  CHAPTER 2   │
│              │   |   │              │
│  ...text...  │   |   │  ...text...  │
│              │   |   │              │
│  [HANGER] ───┼───┼───┼→ [HOOK]     │
│  (Ending)    │   |   │  (Opening)   │
└──────────────┘   |   └──────────────┘
                   |
            "The Handshake"
```

### Output

Add handshake findings to scratchpad:

```markdown
## Handshake Check Results

### Ch1 → Ch2 Transition
- **Smoothness:** 4/5
- **Bridge concept:** Evolution from colonial lodges to revolutionary participation
- **Issue:** Ch2 opening re-explains "Entered Apprentice" degree unnecessarily
- **Fix:** Remove redundant explanation in Ch2, add cross-reference instead

### Ch2 → Ch3 Transition
- **Smoothness:** 2/5 ⚠️
- **Issue:** Ch2 ends conclusively; Ch3 opens on entirely new topic
- **Fix:** Add forward reference at end of Ch2; soften Ch3 opening to callback
```

### Git Checkpoint

```bash
git add projects/books/BOOK_xxx/revised/ projects/books/BOOK_xxx/scratchpad.md
git commit -m "Phase 7.5: Handshake checks complete, transitions improved"
```

---

## Phase 8: Polish

### Purpose
Final refinement of near-complete chapters. Subtle improvements only.

### Step 8.1: Verify Ready for Polish

**Tell Claude:**

> "Is Chapter 1 ready for polish? Check:
> - All scratchpad issues resolved
> - No structural changes needed
> - Citations complete"

### Step 8.2: Polish Chapter

**Two approaches available:**

**Option A: Claude Code applies polish suggestions (Default)**

> "Polish Chapter 1 using the `micro_enrichment` prompt.
>
> **Parameters:**
> - Domain: historical
> - Preserve tone: narrative
> - Max changes per paragraph: 2
>
> Make only subtle improvements. This chapter is nearly publication-ready."

```bash
./rdf polish chapter_01.md --preset academic
# Returns suggestions for Claude Code to apply
```

**Option B: Delegate polishing to internal model**

For bulk polishing where you want the CLI to handle rewrites:

```bash
./rdf polish chapter_01.md --preset academic --use-internal-model
# CLI rewrites text using internal LLM
```

**See:** [prompts/polish/](../prompts/polish/)

### Step 8.3: Cross-Chapter Cohesion (After ALL Chapters)

**Only after all chapters are polished:**

> "Run cohesion check using the `cohesion_check` prompt.
>
> **Track terminology:** lodge, ritual, initiation, brother, craft
> **Track themes:** secrecy evolution, tradition vs reform
> **Check:** Date consistency, name spelling, cross-references"

**See:** [prompts/polish/cohesion_check.md](../prompts/polish/cohesion_check.md)

### Output Structure

```
projects/books/BOOK_xxx/
├── polished/
│   ├── chapter_01.md
│   ├── chapter_02.md
│   └── ...
├── cohesion_report.md        ← Cross-chapter issues found
```

### Git Checkpoint

```bash
git add projects/books/BOOK_xxx/polished/ projects/books/BOOK_xxx/cohesion_report.md
git commit -m "Phase 8: All chapters polished, cohesion checked"
```

---

## Phase 9: Front Matter & Final Compilation

### Purpose
Create the packaging for the book (Introduction, TOC) and handle technical rendering (Bibliography, Endnotes, Page Numbers).

**Key Insight:** The Introduction should be written **last**, after the body is complete—it needs to preview arguments you *actually made*, not arguments you *planned to make*.

---

### Step 9.1: The "Bookend" Writing (Introduction & Conclusion Review)

Now that the body is finished, write the pieces that frame it.

**Tell Claude:**

> "Review the final polished drafts of all chapters. Based on what we actually wrote, draft the **Introduction**.
>
> **Goal:** Preview the specific arguments we ended up making (not the original outline—the final reality).
> **Hook:** Align with the tone of Chapter 1's opening.
> **Promise:** Only promise what the book actually delivers.
>
> Also review the **Conclusion**: Does it synthesize what we actually argued, or does it summarize the outline?"

**Why write the Introduction last:** The Introduction promises what the book will deliver. If you write it first, you're promising based on an outline. Writing it last ensures you're promising what you *actually delivered*.

---

### Step 9.2: Technical Front Matter (No Page Numbers!)

**Tell Claude:**

> "Generate the Front Matter **content only** (metadata, not layout):
>
> - **Title Page text:** Title, subtitle, author
> - **Dedication:** [Placeholder or draft]
> - **Preface/Acknowledgments:** [Placeholder or draft]
> - **Table of Contents Structure:** List chapter titles in order
>
> **IMPORTANT:** Do NOT assign page numbers to the TOC. Page numbers will be generated during PDF/DOCX export when the document is physically laid out."

**Why no page numbers:** An LLM works in tokens, not physical pages. Page numbers depend on font size, margins, and paper size—information only available during final rendering. Any page numbers Claude generates would be hallucinated.

**TOC Output Format:**

```markdown
## Table of Contents

1. Introduction
2. Colonial Origins
3. The Founding Generation
4. [...]
8. Conclusion

Appendix A: [...]
Bibliography
Index
```

Page numbers are inserted by Pandoc/Word during export.

---

### Step 9.3: Citation Audit & Bibliography

**Step 9.3.1: Citation Syntax Audit**

**Tell Claude:**

> "Audit the citation syntax across all chapters:
>
> 1. Verify every citation uses `[^n]` Markdown syntax
> 2. Flag any HTML `<sup>` tags (should not exist)
> 3. Flag any parenthetical citations like `(Smith 2023)` (should not exist)
> 4. Verify each chapter has reference definitions at the bottom
> 5. Report any orphaned citations (referenced but not defined) or orphaned definitions (defined but not referenced)"

**Step 9.3.2: Generate Bibliography (CLI)**

**Tell Claude:**

> "Generate the bibliography using the CLI."

**What Claude will do:**

```bash
./rdf book projects/books/BOOK_xxx/outline.yaml --compile --include-bibliography
```

This generates `compiled/bibliography.md` from the sources actually cited in your chapters.

**Why CLI-based:** The book command extracts citations from your ingested document metadata, preventing LLM hallucination of publication dates, publishers, or page ranges.

**Step 9.3.3: Export Bibliography in Academic Formats**

For integration with reference managers (Zotero, Mendeley, EndNote), export the bibliography in standard formats:

```bash
# Export as BibTeX (for LaTeX and reference managers)
./rdf export bibliography --project BOOK_xxx --format bibtex --output refs.bib

# Export as RIS (for EndNote, Zotero, Mendeley)
./rdf export bibliography --project BOOK_xxx --format ris --output refs.ris

# Export as CSL-JSON (for Pandoc and Zotero)
./rdf export bibliography --project BOOK_xxx --format csl-json --output refs.json

# Validate sources before export (check for missing fields)
./rdf export bibliography --project BOOK_xxx --validate
```

**BibTeX output example:**
```bibtex
@book{steiner1894philosophy,
  author = {Steiner, Rudolf},
  title = {Philosophy of Freedom},
  year = {1894},
  publisher = {Rudolf Steiner Press},
  note = {RDF Document ID: DOC_023}
}
```

---

### Step 9.4: The "Editor-in-Chief" Review

**Purpose:** A high-level **developmental edit** of the full manuscript. This is NOT proofreading—it's evaluating the book as a unified work.

**Tell Claude:**

> "Act as an Editor-in-Chief. Read the full compiled manuscript (or the chapter briefs sequence if the full text exceeds context limits).
>
> **Evaluate:**
>
> 1. **The Arc:** Does the book fulfill the promise of the Introduction? Does the argument build progressively?
>
> 2. **Pacing:** Are there chapters that feel sluggish compared to others? Does momentum flag in the middle?
>
> 3. **The Conclusion:** Does the final chapter *synthesize* the arguments of Chapters 1-N, or merely *summarize* them? (Synthesis is deeper—it should offer new insight from the combined journey.)
>
> 4. **Promises Kept:** Review the Introduction. Does every promise made get fulfilled somewhere in the book?
>
> 5. **Narrative Cohesion:** Does this read as one continuous work or as separate essays bound together?
>
> **Provide:** A 'Developmental Notes' report with specific recommendations."

**Developmental Notes Template:**

```markdown
# Editor-in-Chief: Developmental Notes

## Overall Assessment
[1-2 paragraph summary of the manuscript's strengths and areas for improvement]

## The Arc
- **Promise in Introduction:** [What the book promises to deliver]
- **Delivery Assessment:** [How well it delivers]
- **Gap:** [Any unfulfilled promises]

## Pacing Analysis
| Chapter | Pacing | Notes |
|---------|--------|-------|
| Ch1 | Good | Strong opening momentum |
| Ch2 | Good | Builds well |
| Ch3 | Slow ⚠️ | Dense material, consider splitting |
| Ch4 | Good | |
| ... | | |

## Conclusion Quality
- **Type:** Summary / Synthesis
- **Assessment:** [Does it offer new insight or just recap?]
- **Recommendation:** [Specific improvement]

## Critical Issues
1. [Issue requiring attention]
2. [Issue requiring attention]

## Minor Issues
- [List of smaller concerns]

## Final Verdict
- [ ] Ready for publication
- [ ] Ready with minor edits
- [ ] Requires significant revision (see Critical Issues)
```

**See:** [prompts/review/book_editor_review.md](../prompts/review/book_editor_review.md)

### Step 9.5: Assemble Manuscript

**Tell Claude:**

> "Assemble all components into the final manuscript structure:
> - Front matter (Title, TOC structure, Preface)
> - Body (All polished chapters in order)
> - Back matter (Bibliography, Appendices, Index placeholder)"

---

### Step 9.6: Final Export & Rendering

This is where technical rendering happens—page numbers, endnotes, and formatting.

**Tell Claude:**

> "Export the final manuscript.
>
> **Input:** Assembled manuscript (front + chapters + back matter)
> **Output format:** PDF and/or DOCX
>
> **Compilation Settings:**
> - **Citation rendering:** Convert `[^n]` to superscript endnotes (collected at end of book)
> - **TOC:** Generate real page numbers now (via Pandoc)
> - **Headers/Footers:** [Author name / Book title / Page number]"

**The Footnotes vs. Endnotes Switch:**

| Setting | Result |
|---------|--------|
| `--endnotes` | All `[^n]` citations collected at end of book |
| `--footnotes` | Citations appear at bottom of each page |
| `--chapter-endnotes` | Citations collected at end of each chapter |

```bash
# Endnotes at back of book (Chicago style typical)
./rdf book projects/books/BOOK_xxx/outline.yaml --compile --endnotes --format pdf

# Footnotes at bottom of page (academic papers)
./rdf book projects/books/BOOK_xxx/outline.yaml --compile --footnotes --format pdf
```

**Note:** This assumes the compilation command uses Pandoc or similar tool that handles physical rendering. The LLM cannot determine page numbers—only the rendering engine can.

---

### Final Structure

```
projects/books/BOOK_xxx/
├── compiled/
│   ├── bibliography.md       ← CLI-generated from database
│   ├── front_matter.md       ← Title, TOC structure (no page numbers)
│   ├── back_matter.md        ← Appendices, index placeholder
│   ├── manuscript.md         ← Complete assembled markdown
│   └── final/
│       ├── BOOK_TITLE.pdf    ← Rendered with real page numbers
│       ├── BOOK_TITLE.docx   ← For further editing
│       └── chapters/
│           ├── ch01_final.md
│           └── ...
```

### Git Checkpoint (Final)

```bash
git add projects/books/BOOK_xxx/compiled/
git commit -m "Phase 9: Final manuscript compiled and exported"
git tag -a "v1.0-manuscript" -m "First complete manuscript"
```

---

## Quick Reference: What to Tell Claude

### Phase 2: Scoping
> "Create a book project about [topic] with [N] chapters. Then show me the outline so I can refine it."

### Phase 3: Research
> "Research [topic] for Chapter [N]. Save to the project research folder."

### Phase 4: Gap Analysis
> "Analyze research gaps for all chapters. Tell me what's missing."

### Phase 5: Synthesis
> "Create a Chapter Brief for Chapter [N] using the research_synthesis prompt."

**For Chapters 2+ (Bridge Strategy):**
> "Update the Brief for Chapter [N]. Add a 'Transition Strategy' section. Context: Chapter [N-1] ends with [summary]. Ensure the opening bridges from that ending."

### Phase 6: Drafting
> "Draft Chapter [N] using the Chapter Brief and the `[prompt_name]` prompt."

**For Chapters 2+ (Context Injection):**
> "Draft Chapter [N]. Context: Here is the last page of Chapter [N-1]: [paste text]. Ensure the opening connects smoothly."

### Phase 7: Revision
> "Revise Chapter [N] using `[prompt_name]`. Check the scratchpad for cross-chapter issues."

### Phase 7.5: Handshake Check
> "Perform a Handshake Check between Chapter [N] and Chapter [N+1]. Analyze the transition and suggest improvements."

**Full audit:**
> "Perform a Connection Audit across all chapter transitions. Rate smoothness and flag jarring transitions."

### Phase 8: Polish
> "Polish Chapter [N] with `micro_enrichment`. Make only subtle changes."

### Phase 9: Front Matter & Compilation

**Write Introduction LAST:**
> "Based on what we actually wrote, draft the Introduction. Only promise what the book actually delivers."

**Technical Front Matter:**
> "Generate Front Matter content. List chapter titles for TOC but do NOT assign page numbers—those come from Pandoc."

**Citation Audit:**
> "Audit citations: verify all use `[^n]` syntax, no `<sup>` tags, no parenthetical citations."

**Bibliography:**
> "Generate bibliography via CLI" → `./rdf book ... --compile --include-bibliography`

**Editor-in-Chief:**
> "Act as an Editor-in-Chief. Evaluate the arc, pacing, and whether the conclusion synthesizes or merely summarizes."

**Final Export:**
> "Export to PDF with endnotes at back of book" → `./rdf book ... --compile --endnotes --format pdf`

---

## The Scratchpad

The `scratchpad.md` file is critical for maintaining consistency across a long project.

### Usage

**Add a note:**
> "Add to scratchpad: '[Note about cross-chapter issue]'"

**Review before revision:**
> "Show me current scratchpad notes relevant to Chapter [N]."

**Mark resolved:**
> "Update scratchpad: Mark issue [X] as resolved."

### Example Content

```markdown
# Project Scratchpad

## Unresolved Issues

### Dates & Facts
- [ ] Philadelphia Lodge founding: Ch1 says 1731, verify for Ch3
- [ ] "Grand Master" title: Did this exist in colonial period? Check for Ch4

### Terminology
- [ ] Decide: "lodge" vs "Lodge" capitalization (currently inconsistent)
- [ ] Standardize: "brother" vs "Brother"

### Cross-References
- [ ] Ch3 references "as discussed in Chapter 1" - verify passage exists

## Resolved Issues
- [x] Washington's Masonic membership date: Confirmed 1752 (DOC_034)
```

---

## Workflow Checkpoints

At each phase transition, Claude should provide:

```markdown
## Checkpoint: [Phase] Complete

### Completed
- [What was done]

### Files Created/Modified
- [List of files]

### Git Status
- [ ] Changes committed
- [ ] Tag created (if milestone)

### Scratchpad Status
- New issues: [N]
- Resolved: [N]
- Pending: [N]

### Ready for Next Phase?
- [ ] Prerequisites met
- [ ] Quality verified

### Issues Requiring User Input
- [Any problems or questions]
```

---

## See Also

- [Writing Project Phase Guide](WRITING_PROJECT_PHASE_GUIDE.md) - **Claude-guided version**: Claude prompts you through each phase
- [Prompt Library](../prompts/README.md) - All available prompts
- [CLI User Guide](CLI_USER_GUIDE.md) - Complete command reference
- [Canonical Workflows](CANONICAL_WORKFLOWS.md) - Standard workflow patterns
