-- =============================================================================
-- MIGRATION 004: V3 Features - Citations, Pinning, and Research Enhancements
-- =============================================================================
-- Implements V3 roadmap features for stability and usability:
--   - Citation freezing (stable citation keys even if metadata changes)
--   - Document pinning (prioritize documents in research)
--   - Gap pinning (mark specific research gaps as important)
--   - Research session tracking (for interactive agent)
--
-- Run with:
--   PGPASSWORD='...' psql -h localhost -U research_dev_user \
--     -d research_dev_db -f database/migrations/004_v3_features.sql
-- =============================================================================

-- -----------------------------------------------------------------------------
-- Citation Freezing: Add frozen citation fields to documents
-- -----------------------------------------------------------------------------
-- Once a citation key is frozen, it remains stable even if metadata changes.
-- This prevents breaking manuscript references when correcting metadata.
-- -----------------------------------------------------------------------------

ALTER TABLE documents
ADD COLUMN IF NOT EXISTS bibtex_key VARCHAR(100);

ALTER TABLE documents
ADD COLUMN IF NOT EXISTS bibtex_key_frozen BOOLEAN DEFAULT FALSE;

ALTER TABLE documents
ADD COLUMN IF NOT EXISTS bibtex_key_frozen_at TIMESTAMP;

-- Index for frozen citations
CREATE INDEX IF NOT EXISTS idx_documents_bibtex_key
ON documents(bibtex_key) WHERE bibtex_key IS NOT NULL;

-- -----------------------------------------------------------------------------
-- Document Pinning: Prioritize documents in research
-- -----------------------------------------------------------------------------
-- Pinned documents receive higher priority in:
--   - Search result ranking
--   - Context selection for synthesis
--   - Research agent source selection
-- -----------------------------------------------------------------------------

ALTER TABLE documents
ADD COLUMN IF NOT EXISTS is_pinned BOOLEAN DEFAULT FALSE;

ALTER TABLE documents
ADD COLUMN IF NOT EXISTS pin_priority INTEGER DEFAULT 0;

ALTER TABLE documents
ADD COLUMN IF NOT EXISTS pinned_at TIMESTAMP;

ALTER TABLE documents
ADD COLUMN IF NOT EXISTS pin_notes TEXT;

-- Index for pinned documents
CREATE INDEX IF NOT EXISTS idx_documents_pinned
ON documents(is_pinned, pin_priority DESC) WHERE is_pinned = TRUE;

-- -----------------------------------------------------------------------------
-- Research Gap Pinning: Mark important gaps for targeted research
-- -----------------------------------------------------------------------------
-- Users can pin specific gaps identified during research to:
--   - Prioritize them for gap-filling
--   - Track which gaps matter most
--   - Ignore irrelevant gaps
-- -----------------------------------------------------------------------------

CREATE TABLE IF NOT EXISTS research_gaps (
    gap_id SERIAL PRIMARY KEY,

    -- Gap identification
    description TEXT NOT NULL,
    suggested_query TEXT,

    -- Source context
    source_session_id VARCHAR(50),  -- Research session that identified this gap
    source_project_id VARCHAR(50),  -- Book/research project
    source_subject TEXT,            -- Subject being researched

    -- User actions
    status VARCHAR(20) DEFAULT 'pending',  -- pending, pinned, ignored, filled
    priority INTEGER DEFAULT 0,            -- Higher = more important
    pinned_at TIMESTAMP,
    filled_at TIMESTAMP,

    -- Tracking
    search_results_count INTEGER DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Indexes for gap management
CREATE INDEX IF NOT EXISTS idx_research_gaps_status
ON research_gaps(status);

CREATE INDEX IF NOT EXISTS idx_research_gaps_project
ON research_gaps(source_project_id) WHERE source_project_id IS NOT NULL;

CREATE INDEX IF NOT EXISTS idx_research_gaps_priority
ON research_gaps(priority DESC) WHERE status = 'pinned';

-- -----------------------------------------------------------------------------
-- Research Sessions: Track interactive research state
-- -----------------------------------------------------------------------------
-- For interactive agent mode with checkpoint-based approval.
-- Sessions can be paused, reviewed, and resumed.
-- -----------------------------------------------------------------------------

CREATE TABLE IF NOT EXISTS research_sessions (
    session_id VARCHAR(50) PRIMARY KEY,

    -- Query and planning
    original_question TEXT NOT NULL,
    sub_queries JSONB,  -- Array of planned sub-queries

    -- State machine
    state VARCHAR(30) DEFAULT 'planning',
    -- States: planning, awaiting_plan_approval, searching,
    --         awaiting_source_review, synthesizing, complete, cancelled

    -- Progress tracking
    current_iteration INTEGER DEFAULT 0,
    max_iterations INTEGER DEFAULT 5,

    -- Results
    results_collected JSONB,  -- Chunks found so far
    gaps_identified JSONB,    -- Gaps found during research
    synthesis TEXT,           -- Final synthesis (when complete)

    -- Cost tracking
    total_input_tokens INTEGER DEFAULT 0,
    total_output_tokens INTEGER DEFAULT 0,
    estimated_cost_usd NUMERIC(10, 6) DEFAULT 0,
    budget_limit_usd NUMERIC(10, 2) DEFAULT 1.00,

    -- Interactive mode
    requires_approval BOOLEAN DEFAULT FALSE,
    last_checkpoint TEXT,     -- What user needs to approve
    approved_steps JSONB,     -- Steps user has approved

    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP,

    -- Project association
    project_id VARCHAR(50)
);

-- Indexes for session management
CREATE INDEX IF NOT EXISTS idx_research_sessions_state
ON research_sessions(state);

CREATE INDEX IF NOT EXISTS idx_research_sessions_project
ON research_sessions(project_id) WHERE project_id IS NOT NULL;

-- -----------------------------------------------------------------------------
-- Concept Relationships: Enhanced for semantic traversal
-- -----------------------------------------------------------------------------
-- Add relationship type metadata for semantic graph queries.
-- Supports queries like "find concepts that support X" or "trace origins of Y".
-- -----------------------------------------------------------------------------

-- Check if concept_relationships exists, create if not
CREATE TABLE IF NOT EXISTS concept_relationships (
    relationship_id SERIAL PRIMARY KEY,
    source_concept_id INTEGER REFERENCES concepts(concept_id) ON DELETE CASCADE,
    target_concept_id INTEGER REFERENCES concepts(concept_id) ON DELETE CASCADE,
    relationship_type VARCHAR(50) NOT NULL DEFAULT 'related',
    weight FLOAT DEFAULT 1.0,
    document_id VARCHAR(50),
    chunk_id VARCHAR(50),
    extraction_method VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    UNIQUE(source_concept_id, target_concept_id, relationship_type)
);

-- Add semantic metadata columns if they don't exist
ALTER TABLE concept_relationships
ADD COLUMN IF NOT EXISTS confidence FLOAT DEFAULT 1.0;

ALTER TABLE concept_relationships
ADD COLUMN IF NOT EXISTS bidirectional BOOLEAN DEFAULT FALSE;

ALTER TABLE concept_relationships
ADD COLUMN IF NOT EXISTS verified BOOLEAN DEFAULT FALSE;

-- Indexes for semantic graph traversal
CREATE INDEX IF NOT EXISTS idx_concept_rel_source
ON concept_relationships(source_concept_id);

CREATE INDEX IF NOT EXISTS idx_concept_rel_target
ON concept_relationships(target_concept_id);

CREATE INDEX IF NOT EXISTS idx_concept_rel_type
ON concept_relationships(relationship_type);

-- Composite index for relationship queries
CREATE INDEX IF NOT EXISTS idx_concept_rel_semantic
ON concept_relationships(source_concept_id, relationship_type, target_concept_id);

-- -----------------------------------------------------------------------------
-- Update trigger for timestamps
-- -----------------------------------------------------------------------------

CREATE OR REPLACE FUNCTION update_updated_at_column()
RETURNS TRIGGER AS $$
BEGIN
    NEW.updated_at = CURRENT_TIMESTAMP;
    RETURN NEW;
END;
$$ language 'plpgsql';

-- Apply to research_gaps
DROP TRIGGER IF EXISTS update_research_gaps_updated_at ON research_gaps;
CREATE TRIGGER update_research_gaps_updated_at
    BEFORE UPDATE ON research_gaps
    FOR EACH ROW
    EXECUTE FUNCTION update_updated_at_column();

-- Apply to research_sessions
DROP TRIGGER IF EXISTS update_research_sessions_updated_at ON research_sessions;
CREATE TRIGGER update_research_sessions_updated_at
    BEFORE UPDATE ON research_sessions
    FOR EACH ROW
    EXECUTE FUNCTION update_updated_at_column();

-- -----------------------------------------------------------------------------
-- Migration complete message
-- -----------------------------------------------------------------------------
DO $$
BEGIN
    RAISE NOTICE 'Migration 004: V3 Features applied successfully';
    RAISE NOTICE '  - Citation freezing columns added to documents';
    RAISE NOTICE '  - Document pinning columns added to documents';
    RAISE NOTICE '  - research_gaps table created';
    RAISE NOTICE '  - research_sessions table created';
    RAISE NOTICE '  - concept_relationships enhanced for semantic traversal';
END $$;
